/*
 * Copyright (c) 2016-2025 Apple, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 *
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this
 * file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 *
 * @APPLE_LICENSE_HEADER_END@
 */

#ifndef vfs_unicode_data_h
#define vfs_unicode_data_h

/*
 * Trie structure
 *
 * The trie (with associated code in normalizeCaseFoldU32Code) provides a way to
 * map a single Unicode code point either to:
 * - a flag indicating that it is invalid (either a noncharacter, or unassigned in
 *   the currently-supported version of Unicode); else
 * - its NFD-normalized form if it has one - which might be a sequence
 *   of up to 4 code points - along with the combining class for the first of those
 *   code points; else
 * - its case-folded form in cases where NFD normalization has no effect; else
 * - its combining class (a number in the range 0..254) if non-zero; else
 * - an indication that none of the above apply.
 *
 * The trie takes advantage of certain aspects of the layout of Unicode code point space,
 * which is divided up as follows (check the ranges and numeric values below for each
 * new Unicode version):
 *
 * 0000-323FF   Mix of assigned characters and non-characters; notable subranges:
 *   0000-007F  ASCII range, characters in this range are never looked up in the trie.
 *              - They have no decompositions, and all have combining class 0.
 *                Some have case foldings.
 *   AC00-D7A3  Hangul syllables, which have algorithmic decompositions (not looked up
 *              in the trie) to conjoining jamo in the range 1100-11C2, and have
 *              combining class 0 (and no case folding!). None of these are the result
 *              of any decomposition or folding.
 *   D7A4-D7FF  Mix of invalid and Hangul jamo. None of these have decompositions,
 *              foldings or nonzero combining class, and none of these are the result
 *              of any decomposition or folding.
 *   D800-DFFF  Surrogate code points. These are not valid code characters and will not
 *              be produced by conversion from UTF-8.
 *   E000-F8FF  Private use characters. These have no decompositions or case foldings,
 *              and all have combining class 0. None of these are the result of any
 *              decomposition or folding.
 *
 * 32400-DFFFF  Invalid
 * E0000-E01FF  Mix of assigned characters and non-characters
 * E0200-EFFFF  Invalid
 * F0000-10FFFF Private use (no decomps or case foldings, combining class 0) except that
 *              FFFFE-FFFFF and 10FFFE-10FFFF are invalid.
 *
 * The ranges 32400-DFFFF and E0200-10FFFF are handled by code.
 * The range E0000-E01FF is handled for trie lookup by shifting it down to follow the
 * range 0000-323FF. Thus the trie handles data lookup for "lookup codepoints" in the
 * range 0000-325FF (18 bits, instead of the full 21-bit range for Unicode codepoints),
 * with the lookup codepoints 32400-325FF corresponding to real code points E0000-E01FF.
 *
 * Note that no decomposition or folding produces a value in the range AC00-F8FF, and
 * no code point in AC00-F8FF has a non-algorithmic decomposition, a case folding, or
 * a non-zero combining class. Thus the trie can use values in this range for special
 * purposes.
 *
 * For case folding we use a simple table of the single-character foldings up through
 * code point 04FF. This is used for both folding of the first char in sequences that
 * result from NFD decompositions, as well as folding oc characters that don't change
 * with NFD decomposition and have simple case-foldings to another single character.
 * For other case-folding situations we use a flag in the normal to  enable additional
 * mappings for case folding.
 *
 * The trie is divided into 3 stages. The first (nfTrieHi) maps the high-order bits
 * (bits 8 and up) of the lookup codepoint to a one of the following trie values or
 * value ranges which specifies certain behavior for the range of lookup codepoint
 * with those high-order bits:
 *   0:                 => every lookup codepoint in the range corresponds to a Unicode
 *                      codepoint that is valid and has no decomposition, case folding,
 *                      or non-zero combining class.
 *   kInvalidCodeFlag:  => every lookup codepoint in the range corresponds to a Unicode
 *                      codepoint that is invalid for the currently-supported version of
 *                      Unicode.
 *   if neither of those, then check trie lookup value & kFlagTestMask for the following:
 *   kHangulMask:       => every lookup codepoint in the range corresponds to a Unicode
 *                      codepoint that is a composed Hangul syllable with an algorithmic
 *                      decomposition. but no case folding or non-zero combining class
 *   kCombClassFlag:    => every lookup codepoint in the range corresponds to a Unicode
 *                      codepoint that has a non-zero combining class, which is specified
 *                      in the low-order 8 bits of the trie value.
 *   Otherwise the high-order 4 bits of the trie value match kNextIndexMask, and the
 *   low-order 12 bits specify an index into the middle trie stage nfTrieMid.
 *
 * Each entry in the second or middle trie stage (nfTrieMid) maps (for a particular
 * combination of high bits) the middle-order bits (bits 4-7) to one of the following
 * trie values or value ranges which specifies certain behavior for the range of lookup
 * codepoint with the specified high-order and middle-order bits:
 *   0:                 (as for the high-order stage above)
 *   kInvalidCodeFlag:  (as for the high-order stage above)
 *   if neither of those, then check trie lookup value & kFlagTestMask for the following:
 *   kHangulMask:       (as for the high-order stage above)
 *   kCombClassFlag:    (as for the high-order stage above)
 *   kInvMaskFlag:      => every lookup codepoint in the range corresponds to a Unicode
 *                      codepoint that should be treated having the trie value either 0
 *                      (no special handling) or kInvalidCodeFlag (invalid). The low-order
 *                      8 bits of the trie value specifies an index into nfU16InvMasks,
 *                      which provides a bitmask indicating which of the 16 codepoints in
 *                      the range should be treated as having trie value kInvalidCodeFlag.
 *   Otherwise the high-order 4 bits of the trie value match kNextIndexMask, and the
 *   low-order 12 bits specify an index into the low trie stage nfTrieLo.
 *
 * Each entry in the third or low trie stage (nfTrieLo) maps (for a particular
 * combination of bits 4 and higher) the low-order bits (bits 0-3) to one of the following
 * trie values which specifies certain behavior for the specific lookup codepoint with the
 * specified high-,  middle-, and low-order bits:
 *   0:                 (as for the high-order stage above)
 *   kInvalidCodeFlag:  (as for the high-order stage above)
 *   if neither of those, then check trie lookup value & kFlagTestMask for the following:
 *   kHangulMask:       (as for the high-order stage above)
 *   kCombClassFlag:    (as for the high-order stage above)
 *   Otherwise the high-order 4 bits of the trie value match one of the following masks,
 *   which determine the result:
 *   kToU16Seq2Mask:    => decomposition/folding to a sequence of 2 BMP codepoints; the
 *                      first has combining class 0. The low order bits of the trie value
 *                      are an index into nfU16Seq2 which provides the sequence.
 *   kToU16Seq3Mask:    => decomposition/folding to a sequence of 3 BMP codepoints; the
 *                       first has combining class 0. The low order bits of the trie value
 *                       are an index into nfU16Seq3 which provides the sequence.
 *   kToU16SeqMiscMask: => mapping to a BMP sequence of length up to 4 in which the first
 *                      character may have a nonzero combining class. The low order bits
 *                      of the trie value are an index into nfU16SeqMisc; the element
 *                      at that index has the sequence length in the low 4 bits and the
 *                      combining class of the first element in the bits 4 and up; the
 *                      following elements are the UNicode codepoints of the sequence.
 *   kToU32CharMask:    => decomposition/folding to a single non-BMP codepoint, with
 *                      combining class 0. The low order bits of the trie value are an
 *                      index into nfU32Char which provides the result character.
 *   kToU32SeqMiscMask: => mapping to a non-BMP sequence of length up to 4 in which the
 *                      first character may have a nonzero combining class. The low order
 *                      bits of the trie value are an index into nfU32SeqMisc; the element
 *                      at that index has the sequence length in the low 4 bits and the
 *                      combining class of the first element in the bits 4 and up; the
 *                      following elements are the UNicode codepoints of the sequence.
 *
 */

enum {
	kU32LowRangeLimit   = 0x32400,
	kU32HiRangeStart    = 0xE0000,
	kU32HiRangeLen      = 0x200,
	kU32HiPUAStart      = 0xF0000,
	kNFTrieHiShift      = 8,
	kNFTrieMidShift     = 4,
	kNFTrieMidMask      = 0x0F,
	kNFTrieLoMask       = 0x0F,
	/* for hi/mid trie stages only: */
	kNextIndexMask      = 0xC000,
	kNextIndexValueMask = 0x0FFF,
	/* for any trie stage */
	kInvalidCodeFlag    = 0xFFFF,
	kFlagTestMask       = 0xFF00, /*  to test for flag */
	kFlagValueMask      = 0x00FF,
	kHangulMask         = 0xAC00, /*  flag to use algorithmic Hangul decomp */
	kCombClassFlag      = 0xAD00, /*  low 8 bits will be comb class */
	/* for mid trie stage only */
	kInvMaskFlag        = 0xAE00, /*  low 8 bits will be index into invalid mask table */
	/* for lo trie stage only */
	kSpecialsStart      = 0xAC00,
	kSpecialsMask       = 0xF000,
	kToU16Seq2Mask      = 0xB000, /*to BFFF low 11 bits are index  (max 758, 10 bits) in table of 2-element u16 sequences; 12th bit is casefold flag */
	kToU16Seq3Mask      = 0xC000, /*to CFFF low 11 bits are index  (max 220,  8 bits) in table of 3-element u16 sequences; 12th bit is casefold flag */
	kToU16SeqMiscMask   = 0xD000, /*to D6FF low 10 bits are offset (max 200,  8 bits) in table of misc u16 sequences (with length/cc in prefix) */
	kToU32CharMask      = 0xE000, /*to EFFF low 11 bits are index  (max 305,  9 bits) in table of u32 chars; 12th bit is casefold flag */
	kToU32SeqMiscMask   = 0xF000, /*to F8FF low  7 bits are offset (max  84,  7 bits) in table of misc u32 sequences (with length/cc in prefix) */
	kSpecialsEnd        = 0xF8FF,

	kToSeqIndexMask     = 0x07FF,
	kToSeqCaseFoldMask  = 0x0800,
	kToSeqMiscIndexMask = 0x03FF,
	kToSeqMiscLenMask   = 0x000F,
	kToSeqMiscCCShift   = 4,
	kNFCSingleCharDecompMax = 4, /* assumed to be at least 3 by code */
	kSimpleCaseFoldLimit = 0x0500, /* characters less than this mostly handled by simple casefold table */
};

/* Start generated data. */
/* hiCount:           806, nfTrieHi      size  1612 */
/* midCount:          153, nfTrieMid     size  4896 */
/* loCount:           448, nfTrieLo      size 14336 */
/* u16InvMasksIndex:  139, nfU16InvMasks size   278 */
/* u16Seq2Index:      773, nfU16Seq2     size  3092 */
/* u16Seq3Index:      222, nfU16Seq3     size  1332 */
/* u16SeqMiscOffset:  198, nfU16SeqMisc  size   396 */
/* u32CharIndex:      954, nfU32Char     size  3816 */
/* u32SeqMiscOffset:  151, nfU32SeqMisc  size   604 */
/* basicCFCount:     1280, nfBasicCF     size  2560 */

static uint16_t nfTrieHi[806] = {
	/*begidx:  uchar */
	/* 0x000: 0x0000 */ 0xC000, 0xC001, 0xC002, 0xC003, 0xC004, 0xC005, 0xC006, 0xC007, 0xC008, 0xC009, 0xC00A, 0xC00B, 0xC00C, 0xC00D, 0xC00E, 0xC00F,
	/* 0x010: 0x1000 */ 0xC010, 0x0000, 0xC011, 0xC012, 0x0000, 0x0000, 0xC013, 0xC014, 0xC015, 0xC016, 0xC017, 0xC018, 0xC019, 0xC01A, 0xC01B, 0xC01C,
	/* 0x020: 0x2000 */ 0xC01D, 0xC01E, 0xC01F, 0xC020, 0xC021, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC022, 0xC023, 0xC024, 0xC025, 0xC026, 0xC027,
	/* 0x030: 0x3000 */ 0xC028, 0xC029, 0xC02A, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x040: 0x4000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x050: 0x5000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x060: 0x6000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x070: 0x7000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x080: 0x8000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x090: 0x9000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x0A0: 0xA000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0xC02B, 0x0000, 0xC02C, 0xC02D, 0xC02E, 0xC02F, 0xC030, 0xC031, 0xAC00, 0xAC00, 0xAC00, 0xAC00,
	/* 0x0B0: 0xB000 */ 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00,
	/* 0x0C0: 0xC000 */ 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00,
	/* 0x0D0: 0xD000 */ 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xC032, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	/* 0x0E0: 0xE000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x0F0: 0xF000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC033, 0xC034, 0xC035, 0x0000, 0xC036, 0xC037, 0xC038,
	/* 0x100:0x10000 */ 0xC039, 0xC03A, 0xC03B, 0xC03C, 0xC03D, 0xC03E, 0x0000, 0xC03F, 0xC040, 0xC041, 0xC042, 0xC043, 0xC044, 0xC045, 0xC046, 0xC047,
	/* 0x110:0x11000 */ 0xC048, 0xC049, 0xC04A, 0xC04B, 0xC04C, 0xC04D, 0xC04E, 0xC04F, 0xC050, 0xC051, 0xC052, 0xC053, 0xC054, 0xC055, 0xC056, 0xC057,
	/* 0x120:0x12000 */ 0x0000, 0x0000, 0x0000, 0xC058, 0xC059, 0xC05A, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xC05B,
	/* 0x130:0x13000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0xC05C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x140:0x14000 */ 0x0000, 0x0000, 0x0000, 0xC05D, 0x0000, 0x0000, 0xC05E, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	/* 0x150:0x15000 */ 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	/* 0x160:0x16000 */ 0xFFFF, 0xC05F, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xC060, 0xC061, 0xFFFF, 0xC062, 0xC063, 0xC064,
	/* 0x170:0x17000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x180:0x18000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC065, 0x0000, 0x0000, 0x0000, 0x0000, 0xC066, 0xC067, 0xFFFF, 0xFFFF,
	/* 0x190:0x19000 */ 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	/* 0x1A0:0x1A000 */ 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xC068,
	/* 0x1B0:0x1B000 */ 0x0000, 0xC069, 0xC06A, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xC06B, 0xFFFF, 0xFFFF, 0xFFFF,
	/* 0x1C0:0x1C000 */ 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xC06C, 0x0000, 0xC06D, 0xC06E,
	/* 0x1D0:0x1D000 */ 0xC06F, 0xC070, 0xC071, 0xC072, 0xC073, 0xC074, 0xC075, 0xC076, 0x0000, 0x0000, 0xC077, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xC078,
	/* 0x1E0:0x1E000 */ 0xC079, 0xC07A, 0xC07B, 0xFFFF, 0xC07C, 0xC07D, 0xFFFF, 0xC07E, 0xC07F, 0xC080, 0xFFFF, 0xFFFF, 0xC081, 0xC082, 0xC083, 0xFFFF,
	/* 0x1F0:0x1F000 */ 0xC084, 0xC085, 0xC086, 0x0000, 0x0000, 0x0000, 0xC087, 0xC088, 0xC089, 0x0000, 0xC08A, 0xC08B, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	/* 0x200:0x20000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x210:0x21000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x220:0x22000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x230:0x23000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x240:0x24000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x250:0x25000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x260:0x26000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x270:0x27000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x280:0x28000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x290:0x29000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2A0:0x2A000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC08C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2B0:0x2B000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC08D, 0xC08E, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2C0:0x2C000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC08F, 0x0000,
	/* 0x2D0:0x2D000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x2E0:0x2E000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC090, 0x0000, 0x0000, 0xC091, 0xFFFF,
	/* 0x2F0:0x2F000 */ 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xC092, 0xC093, 0xC094, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	/* 0x300:0x30000 */ 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x310:0x31000 */ 0x0000, 0x0000, 0x0000, 0xC095, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	/* 0x320:0x32000 */ 0x0000, 0x0000, 0x0000, 0xC096, 0xC097, 0xC098
};
static uint16_t nfTrieMid[153][16] = {
	/* 0x000 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC000, 0xC001, 0xC002, 0xC003 },
	/* 0x001 */ { 0xC004, 0xC005, 0xC006, 0xC007, 0xC008, 0xC009, 0xC00A, 0xC00B, 0x0000, 0x0000, 0xC00C, 0xC00D, 0xC00E, 0xC00F, 0xC010, 0xC011 },
	/* 0x002 */ { 0xC012, 0xC013, 0xC014, 0xC015, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x003 */ { 0xADE6, 0xC016, 0xC017, 0xC018, 0xC019, 0xC01A, 0xC01B, 0xC01C, 0xC01D, 0xC01E, 0xC01F, 0xC020, 0xC021, 0xC022, 0x0000, 0x0000 },
	/* 0x004 */ { 0xC023, 0xC024, 0x0000, 0xC025, 0x0000, 0xC026, 0x0000, 0xC027, 0xC028, 0x0000, 0x0000, 0x0000, 0xC029, 0xC02A, 0xC02B, 0xC02C },
	/* 0x005 */ { 0xC02D, 0xC02E, 0xC02F, 0xC030, 0xC031, 0xC032, 0x0000, 0x0000, 0xC033, 0xC034, 0xC035, 0xC036, 0xC037, 0x0000, 0xAE00, 0xAE01 },
	/* 0x006 */ { 0x0000, 0xC038, 0xC039, 0x0000, 0xC03A, 0xC03B, 0x0000, 0xC03C, 0x0000, 0x0000, 0x0000, 0x0000, 0xC03D, 0xC03E, 0xC03F, 0x0000 },
	/* 0x007 */ { 0xAE02, 0xC040, 0x0000, 0xC041, 0xC042, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE03, 0x0000, 0x0000, 0xC043, 0xC044 },
	/* 0x008 */ { 0x0000, 0xC045, 0xC046, 0xAE04, 0x0000, 0xC047, 0xAE05, 0x0000, 0xAE04, 0xC048, 0x0000, 0x0000, 0xC049, 0xC04A, 0xC04B, 0xC04C },
	/* 0x009 */ { 0x0000, 0x0000, 0xC04D, 0xC04E, 0xC04F, 0xC050, 0x0000, 0x0000, 0xAE06, 0xAE07, 0xAE08, 0xC051, 0xC052, 0xC053, 0xAE09, 0xC054 },
	/* 0x00A */ { 0xAE0A, 0xAE07, 0xAE08, 0xC055, 0xC056, 0xC057, 0xAE0B, 0xAE0C, 0xAE0D, 0xAE0E, 0xAE08, 0xC058, 0xC059, 0xAE0F, 0xAE09, 0xAE10 },
	/* 0x00B */ { 0xAE11, 0xAE07, 0xAE08, 0xC05A, 0xC05B, 0xC05C, 0xAE09, 0xAE12, 0xAE13, 0xC05D, 0xAE14, 0xAE15, 0xC05E, 0xAE16, 0xAE0B, 0xAE05 },
	/* 0x00C */ { 0xAE17, 0xAE18, 0xAE08, 0xC05F, 0xC060, 0xC061, 0xAE09, 0xAE19, 0xAE17, 0xAE18, 0xAE08, 0xC062, 0xC063, 0xAE1A, 0xAE09, 0xAE1B },
	/* 0x00D */ { 0xAE17, 0xAE18, 0x0000, 0xC064, 0xC065, 0xAE1C, 0xAE09, 0x0000, 0xAE1D, 0xAE1E, 0x0000, 0xAE1F, 0xC066, 0xC067, 0xAE0B, 0xAE20 },
	/* 0x00E */ { 0xAE21, 0x0000, 0x0000, 0xC068, 0xC069, 0xAE22, 0xFFFF, 0xFFFF, 0xAE23, 0x0000, 0xAE24, 0xC06A, 0xC06B, 0xAE25, 0xFFFF, 0xFFFF },
	/* 0x00F */ { 0x0000, 0xC06C, 0x0000, 0xC06D, 0xC06E, 0xC06F, 0xC070, 0xC071, 0xC072, 0xC073, 0xC074, 0xC075, 0xC076, 0xAE05, 0xFFFF, 0xFFFF },
	/* 0x010 */ { 0x0000, 0x0000, 0xC077, 0xC078, 0x0000, 0x0000, 0x0000, 0x0000, 0xC079, 0x0000, 0xC07A, 0xC07B, 0xC07C, 0x0000, 0x0000, 0x0000 },
	/* 0x011 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAE26, 0xAE27, 0x0000, 0x0000, 0xAE26, 0x0000, 0x0000, 0xAE28, 0xAE29, 0xAE2A, 0x0000, 0x0000 },
	/* 0x012 */ { 0x0000, 0xAE29, 0x0000, 0x0000, 0x0000, 0xC07D, 0x0000, 0xAE2B, 0x0000, 0xAE2C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC07E },
	/* 0x013 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2B, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2D },
	/* 0x014 */ { 0x0000, 0xC07F, 0x0000, 0xC080, 0x0000, 0xAE2E, 0xAE17, 0xAE2F, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC081, 0xAE2C, 0xAE2C },
	/* 0x015 */ { 0x0000, 0xAE2C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2D, 0x0000, 0x0000, 0xC082, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE30 },
	/* 0x016 */ { 0x0000, 0xAE04, 0xAE22, 0xC083, 0xAE31, 0x0000, 0xAE32, 0xAE01, 0x0000, 0x0000, 0xAE22, 0x0000, 0xAE2C, 0xAE33, 0x0000, 0x0000 },
	/* 0x017 */ { 0x0000, 0xC084, 0x0000, 0x0000, 0x0000, 0xAE04, 0xC085, 0xC086, 0xAE2C, 0xAE2C, 0xAE32, 0xC087, 0xC088, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x018 */ { 0xC089, 0xC08A, 0x0000, 0xC08B, 0xC08C, 0x0000, 0xC08D, 0xC08E, 0x0000, 0x0000, 0xC08F, 0x0000, 0x0000, 0x0000, 0xC090, 0xC091 },
	/* 0x019 */ { 0x0000, 0x0000, 0x0000, 0xC092, 0xAE34, 0x0000, 0x0000, 0x0000, 0xC093, 0xC094, 0xC095, 0xC096, 0xAE12, 0xC097, 0xC098, 0xC099 },
	/* 0x01A */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC09A, 0xC09B, 0xADE6, 0xC09C },
	/* 0x01B */ { 0xC09D, 0xC09E, 0xC09F, 0xC0A0, 0xC0A1, 0xC0A2, 0xC0A3, 0xC0A4, 0xC0A5, 0xC0A6, 0xC0A7, 0xC0A8, 0xC0A9, 0xC0AA, 0xC0AB, 0xC0AC },
	/* 0x01C */ { 0xC0AD, 0xC0AE, 0xC0AF, 0xC0B0, 0xC0B1, 0xC0B2, 0xC0B3, 0xC0B4, 0xC0B5, 0xC0B6, 0xC0B7, 0xC0B8, 0xC0B9, 0xC0BA, 0xC0BB, 0xC0BC },
	/* 0x01D */ { 0xC0BD, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE35, 0xAE36, 0xAE04, 0xAE2B, 0x0000, 0x0000, 0xAE0F, 0xC0BE, 0xC0BF, 0xC0C0 },
	/* 0x01E */ { 0x0000, 0x0000, 0xC0C1, 0xC0C2, 0x0000, 0x0000, 0xC0C3, 0x0000, 0xC0C4, 0xC0C5, 0xC0C6, 0x0000, 0xC0C7, 0x0000, 0x0000, 0x0000 },
	/* 0x01F */ { 0xC0C8, 0x0000, 0xC0C9, 0x0000, 0xC0CA, 0x0000, 0xC0CB, 0xC0CC, 0xC0CD, 0x0000, 0xC0CE, 0x0000, 0x0000, 0x0000, 0xC0CF, 0x0000 },
	/* 0x020 */ { 0x0000, 0x0000, 0xC0D0, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x021 */ { 0x0000, 0x0000, 0xAE2C, 0xFFFF, 0xAE05, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC0D1, 0xC0D2, 0x0000, 0x0000, 0x0000 },
	/* 0x022 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC0D3, 0x0000, 0x0000 },
	/* 0x023 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE09, 0x0000, 0xAE37, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x024 */ { 0xC0D4, 0xC0D5, 0xC0D6, 0x0000, 0x0000, 0x0000, 0xC0D7, 0xC0D8, 0xC0D9, 0xC0DA, 0xC0DB, 0xC0DC, 0xC0DD, 0xC0DE, 0xC0DF, 0xC0E0 },
	/* 0x025 */ { 0x0000, 0x0000, 0xAE38, 0x0000, 0x0000, 0x0000, 0xAE39, 0xC0E1, 0x0000, 0xAE0C, 0xAE3A, 0xAE3A, 0xAE3A, 0xAE3A, 0xADE6, 0xADE6 },
	/* 0x026 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE32, 0xFFFF, 0xFFFF, 0x0000, 0xAE3B, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2E },
	/* 0x027 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE30, 0xFFFF, 0x0000 },
	/* 0x028 */ { 0x0000, 0x0000, 0xC0E2, 0x0000, 0xC0E3, 0xC0E4, 0xC0E5, 0xC0E6, 0x0000, 0xC0E7, 0xC0E8, 0xC0E9, 0xC0EA, 0xC0EB, 0x0000, 0xC0EC },
	/* 0x029 */ { 0xAE3C, 0x0000, 0x0000, 0xAE21, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE04, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE3D, 0x0000 },
	/* 0x02A */ { 0x0000, 0xAE04, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x02B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2B, 0x0000, 0x0000, 0x0000, 0xAE0C, 0x0000, 0x0000, 0x0000 },
	/* 0x02C */ { 0x0000, 0x0000, 0xAE22, 0xFFFF, 0xC0ED, 0xC0EE, 0xC0EF, 0xC0F0, 0xC0F1, 0xC0F2, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC0F3 },
	/* 0x02D */ { 0x0000, 0x0000, 0xC0F4, 0xC0F5, 0xC0F6, 0xC0F7, 0xC0F8, 0xC0F9, 0xC0FA, 0xC0FB, 0xC0FC, 0xC0FD, 0xC0FE, 0xC0FF, 0xFFFF, 0xC100 },
	/* 0x02E */ { 0xC101, 0x0000, 0xC102, 0xAE2C, 0x0000, 0x0000, 0x0000, 0xAE12, 0x0000, 0x0000, 0x0000, 0x0000, 0xC103, 0xAE2C, 0xADE6, 0xC104 },
	/* 0x02F */ { 0x0000, 0x0000, 0xC105, 0x0000, 0x0000, 0xC106, 0x0000, 0xAE2B, 0x0000, 0x0000, 0x0000, 0xC107, 0xC108, 0xAE15, 0x0000, 0xAE04 },
	/* 0x030 */ { 0x0000, 0x0000, 0x0000, 0xAE0C, 0xAE32, 0xAE25, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xC109, 0xC10A, 0xAE3E, 0x0000, 0xC10B },
	/* 0x031 */ { 0xAE3F, 0xAE40, 0xAE3A, 0x0000, 0x0000, 0x0000, 0xAE22, 0xC10C, 0xC10D, 0xC10E, 0xC10F, 0xC110, 0x0000, 0x0000, 0xC111, 0xAE2C },
	/* 0x032 */ { 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xC112, 0x0000, 0xAE41, 0x0000, 0x0000, 0xAE22 },
	/* 0x033 */ { 0xC113, 0xC114, 0xC115, 0xC116, 0xC117, 0xC118, 0xC119, 0xC11A, 0xC11B, 0xC11C, 0xC11D, 0xC11E, 0xC11F, 0xC120, 0xC121, 0xC122 },
	/* 0x034 */ { 0xC123, 0xC124, 0xC125, 0xC126, 0xC127, 0xC128, 0xC129, 0xC12A, 0xC12B, 0xC12C, 0xC12D, 0xC12E, 0xC12F, 0xC130, 0xFFFF, 0xFFFF },
	/* 0x035 */ { 0xC131, 0xC132, 0xC133, 0xC134, 0xC135, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE42, 0xAE43, 0x0000, 0x0000 },
	/* 0x036 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE44, 0x0000, 0x0000, 0xAE39, 0xFFFF, 0xFFFF, 0x0000 },
	/* 0x037 */ { 0x0000, 0xAE2C, 0xC136, 0x0000, 0x0000, 0xAE45, 0xAE46, 0xAE35, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE47 },
	/* 0x038 */ { 0xAE21, 0x0000, 0xC137, 0xC138, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE04, 0xAE48, 0xAE49, 0xAE3A, 0xAE4A },
	/* 0x039 */ { 0xAE4B, 0x0000, 0xAE2A, 0xAE4C, 0xAE32, 0xAE32, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE05 },
	/* 0x03A */ { 0xAE4D, 0x0000, 0x0000, 0xAE4E, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE04, 0xAE2B, 0xAE0F, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xC139 },
	/* 0x03B */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xAE2B, 0x0000, 0x0000, 0x0000, 0xAE0F, 0xC13A, 0xAE22 },
	/* 0x03C */ { 0x0000, 0x0000, 0xAE4F, 0x0000, 0xAE05, 0x0000, 0x0000, 0xC13B, 0x0000, 0xAE02, 0x0000, 0x0000, 0xAE50, 0xAE30, 0xFFFF, 0xFFFF },
	/* 0x03D */ { 0xC13C, 0xC13D, 0xC13E, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE32, 0xAE2C, 0xC13F, 0xC140, 0xC141, 0x0000, 0xAE22 },
	/* 0x03E */ { 0x0000, 0x0000, 0xAE12, 0x0000, 0x0000, 0x0000, 0xAE51, 0xC142, 0xC143, 0xC144, 0xAE0E, 0xAE52, 0xC145, 0x0000, 0xC146, 0xAE2E },
	/* 0x03F */ { 0x0000, 0x0000, 0x0000, 0xAE0C, 0x0000, 0xAE30, 0xAE12, 0xFFFF, 0xAE37, 0x0000, 0x0000, 0xAE53, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x040 */ { 0xAE54, 0x0000, 0x0000, 0xAE55, 0x0000, 0xAE37, 0x0000, 0x0000, 0x0000, 0xAE04, 0xAE19, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xAE56 },
	/* 0x041 */ { 0x0000, 0xAE57, 0x0000, 0xAE58, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xAE59, 0x0000, 0xAE44, 0x0000, 0x0000 },
	/* 0x042 */ { 0xC147, 0xAE5A, 0x0000, 0xC148, 0xAE2D, 0xAE2D, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xC149, 0xAE0C },
	/* 0x043 */ { 0x0000, 0x0000, 0x0000, 0xAE5B, 0x0000, 0xAE5C, 0x0000, 0xAE5D, 0x0000, 0xAE5E, 0xAE5F, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x044 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2D, 0xFFFF, 0xFFFF, 0xFFFF, 0xC14A, 0xC14B, 0xC14C, 0xC14D, 0x0000, 0x0000, 0x0000, 0xAE60 },
	/* 0x045 */ { 0x0000, 0x0000, 0xC14E, 0xAE2C, 0x0000, 0xC14F, 0xC150, 0x0000, 0xAE61, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x046 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xAE04, 0x0000, 0x0000, 0xC151, 0xAE03, 0xAE20, 0xFFFF, 0xFFFF, 0xC152 },
	/* 0x047 */ { 0x0000, 0x0000, 0xAE12, 0x0000, 0xC153, 0xC154, 0xFFFF, 0x0000, 0xC155, 0xFFFF, 0xFFFF, 0x0000, 0xAE22, 0xFFFF, 0x0000, 0xAE0C },
	/* 0x048 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xC156, 0xAE44, 0x0000, 0xC157, 0x0000, 0xC158, 0xC159, 0xC15A, 0xAE62, 0x0000, 0xAE2D, 0xAE2C },
	/* 0x049 */ { 0xC15B, 0x0000, 0xC15C, 0xC15D, 0xAE12, 0x0000, 0x0000, 0xC15E, 0x0000, 0x0000, 0x0000, 0x0000, 0xC15F, 0x0000, 0xAE21, 0xAE01 },
	/* 0x04A */ { 0x0000, 0xAE0E, 0x0000, 0xC160, 0xAE03, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE63, 0xAE02, 0xAE2C, 0x0000, 0x0000, 0x0000, 0xC161, 0xAE2C },
	/* 0x04B */ { 0xAE06, 0xAE07, 0xAE08, 0xC162, 0xC163, 0xAE64, 0xC164, 0xC165, 0xC166, 0xC167, 0x0000, 0xAE37, 0xC168, 0xC169, 0xAE65, 0xFFFF },
	/* 0x04C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xC16A, 0xC16B, 0xAE03, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xC16C, 0xC16D, 0xAE2C, 0xFFFF, 0xFFFF },
	/* 0x04D */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xC16E, 0xC16F, 0xAE32, 0xFFFF, 0xFFFF },
	/* 0x04E */ { 0x0000, 0x0000, 0x0000, 0xC170, 0xAE01, 0xAE2C, 0xAE2B, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xC171, 0xAE2C, 0x0000, 0xAE2E, 0xFFFF },
	/* 0x04F */ { 0x0000, 0xAE66, 0xC172, 0x0000, 0xAE0C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x050 */ { 0x0000, 0x0000, 0x0000, 0xC173, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xC174, 0xC175, 0x0000, 0x0000, 0x0000, 0xAE67 },
	/* 0x051 */ { 0xAE68, 0xAE69, 0x0000, 0xC176, 0xC177, 0xAE2C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE6A, 0x0000, 0x0000, 0xAE6A, 0xC178, 0xFFFF },
	/* 0x052 */ { 0x0000, 0x0000, 0x0000, 0xC179, 0xC17A, 0x0000, 0x0000, 0x0000, 0x0000, 0xC17B, 0xAE42, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2D },
	/* 0x053 */ { 0xAE2C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xAE03, 0xAE2C },
	/* 0x054 */ { 0xAE08, 0x0000, 0x0000, 0xC17C, 0xAE30, 0x0000, 0xAE2B, 0x0000, 0x0000, 0xAE44, 0xAE6B, 0xAE0C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x055 */ { 0xAE6C, 0x0000, 0x0000, 0xAE6D, 0xC17D, 0xAE2C, 0xAE6E, 0x0000, 0xAE04, 0xC17E, 0xAE2C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x056 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xAE2D },
	/* 0x057 */ { 0x0000, 0xAE18, 0x0000, 0xAE33, 0xC17F, 0xAE05, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE0F, 0x0000, 0x0000, 0x0000, 0xAE6F },
	/* 0x058 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x059 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE04, 0xAE01, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x05A */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2E, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x05B */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE42 },
	/* 0x05C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE30, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x05D */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE05 },
	/* 0x05E */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAE0C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x05F */ { 0x0000, 0x0000, 0xC180, 0xAE2C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x060 */ { 0x0000, 0x0000, 0x0000, 0xAE2D, 0x0000, 0xAE04, 0xAE15, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE04, 0xAE2C, 0x0000, 0xAE32, 0xC181 },
	/* 0x061 */ { 0x0000, 0x0000, 0x0000, 0xC182, 0xAE30, 0xAE3B, 0xAE0E, 0xAE70, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x062 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xC183, 0xAE2C, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x063 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xC184, 0xC185, 0x0000, 0x0000, 0x0000, 0xAE05, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x064 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAE00, 0x0000, 0x0000, 0x0000, 0xAE39, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE01, 0xC186 },
	/* 0x065 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE12 },
	/* 0x066 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE30, 0xFFFF, 0xAE71 },
	/* 0x067 */ { 0xAE2D, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x068 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE72 },
	/* 0x069 */ { 0x0000, 0x0000, 0xAE42, 0xAE73, 0xFFFF, 0xAE74, 0xAE75, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x06A */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE22 },
	/* 0x06B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE05, 0xAE2B, 0xAE2D, 0xC187, 0xAE2E, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x06C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2C },
	/* 0x06D */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2E, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x06E */ { 0x0000, 0x0000, 0xAE32, 0x0000, 0xAE0C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2E, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x06F */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE30 },
	/* 0x070 */ { 0x0000, 0x0000, 0xAE76, 0x0000, 0x0000, 0xC188, 0xC189, 0xC18A, 0xC18B, 0x0000, 0xC18C, 0xC18D, 0xC18E, 0x0000, 0xAE05, 0xFFFF },
	/* 0x071 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xC18F, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xAE2E, 0x0000, 0xAE2E },
	/* 0x072 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE0C, 0x0000, 0xAE2D, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x073 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE35, 0x0000, 0x0000, 0x0000, 0xAE17, 0xAE77, 0xAE78, 0xAE79, 0x0000, 0x0000, 0x0000 },
	/* 0x074 */ { 0xAE7A, 0xAE7B, 0x0000, 0xAE7C, 0xAE7D, 0xAE18, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x075 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE5C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x076 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE7E, 0x0000, 0x0000, 0x0000 },
	/* 0x077 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE22, 0xAE3E, 0xAE21, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x078 */ { 0x0000, 0xAE04, 0xAE7F, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x079 */ { 0xC190, 0xC191, 0xC192, 0x0000, 0x0000, 0x0000, 0xAE32, 0xFFFF, 0xC193, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x07A */ { 0x0000, 0x0000, 0xAE2B, 0xC194, 0xAE15, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x07B */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xC195, 0xFFFF, 0x0000, 0x0000, 0xC196, 0xAE58 },
	/* 0x07C */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xC197, 0xAE2C },
	/* 0x07D */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xC198, 0xAE00 },
	/* 0x07E */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE80, 0xAE04 },
	/* 0x07F */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE81, 0xC199, 0xFFFF, 0xFFFF },
	/* 0x080 */ { 0xC19A, 0xC19B, 0xC19C, 0x0000, 0xC19D, 0xAE15, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x081 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE21, 0x0000, 0x0000, 0x0000, 0xAE01, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x082 */ { 0xAE21, 0x0000, 0x0000, 0xAE32, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x083 */ { 0xAE79, 0x0000, 0xAE82, 0xAE83, 0xAE84, 0xAE85, 0xAE86, 0xAE87, 0xAE3B, 0xAE22, 0xAE88, 0xAE22, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE03 },
	/* 0x084 */ { 0x0000, 0x0000, 0xAE22, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2E, 0xAE04, 0xAE21, 0xAE21, 0xAE21, 0x0000, 0xAE30 },
	/* 0x085 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE32, 0xFFFF, 0xFFFF, 0xFFFF, 0xAE0B, 0x0000 },
	/* 0x086 */ { 0xAE42, 0x0000, 0x0000, 0xAE22, 0xAE2D, 0xAE03, 0xAE30, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x087 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE59, 0xAE2B, 0xAE2B },
	/* 0x088 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE41, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2C, 0xAE22, 0xAE0F },
	/* 0x089 */ { 0xAE22, 0x0000, 0x0000, 0x0000, 0xAE12, 0xAE2C, 0x0000, 0x0000, 0xAE12, 0x0000, 0xAE32, 0xAE22, 0xAE03, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x08A */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2E, 0xAE32, 0xAE2B, 0xAE58, 0x0000, 0x0000, 0x0000, 0xAE89, 0xAE47, 0xAE2C, 0xAE2D },
	/* 0x08B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE45, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE2C },
	/* 0x08C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF },
	/* 0x08D */ { 0x0000, 0x0000, 0x0000, 0xAE2C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x08E */ { 0x0000, 0xAE32, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x08F */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE03, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x090 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE0F, 0x0000 },
	/* 0x091 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAE32, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x092 */ { 0xC19E, 0xC19F, 0xC1A0, 0xC1A1, 0xC1A2, 0xC1A3, 0xC1A4, 0xC1A5, 0xC1A6, 0xC1A7, 0xC1A8, 0xC1A9, 0xC1AA, 0xC1AB, 0xC1AC, 0xC1AD },
	/* 0x093 */ { 0xC1AE, 0xC1AF, 0xC1B0, 0xC1B1, 0xC1B2, 0xC1B3, 0xC1B4, 0xC1B5, 0xC1B6, 0xC1B7, 0xC1B8, 0xC1B9, 0xC1BA, 0xC1BB, 0xC1BC, 0xC1BD },
	/* 0x094 */ { 0xC1BE, 0xC1BF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x095 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAE05, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x096 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x097 */ { 0xAE8A, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x098 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF }
};
static uint16_t nfTrieLo[448][16] = {
	/* 0x000 */ { 0xB000, 0xB001, 0xB002, 0xB003, 0xB004, 0xB005, 0x0000, 0xB006, 0xB007, 0xB008, 0xB009, 0xB00A, 0xB00B, 0xB00C, 0xB00D, 0xB00E },
	/* 0x001 */ { 0x0000, 0xB00F, 0xB010, 0xB011, 0xB012, 0xB013, 0xB014, 0x0000, 0x0000, 0xB015, 0xB016, 0xB017, 0xB018, 0xB019, 0x0000, 0xB81A },
	/* 0x002 */ { 0xB01B, 0xB01C, 0xB01D, 0xB01E, 0xB01F, 0xB020, 0x0000, 0xB021, 0xB022, 0xB023, 0xB024, 0xB025, 0xB026, 0xB027, 0xB028, 0xB029 },
	/* 0x003 */ { 0x0000, 0xB02A, 0xB02B, 0xB02C, 0xB02D, 0xB02E, 0xB02F, 0x0000, 0x0000, 0xB030, 0xB031, 0xB032, 0xB033, 0xB034, 0x0000, 0xB035 },
	/* 0x004 */ { 0xB036, 0xB037, 0xB038, 0xB039, 0xB03A, 0xB03B, 0xB03C, 0xB03D, 0xB03E, 0xB03F, 0xB040, 0xB041, 0xB042, 0xB043, 0xB044, 0xB045 },
	/* 0x005 */ { 0x0000, 0x0000, 0xB046, 0xB047, 0xB048, 0xB049, 0xB04A, 0xB04B, 0xB04C, 0xB04D, 0xB04E, 0xB04F, 0xB050, 0xB051, 0xB052, 0xB053 },
	/* 0x006 */ { 0xB054, 0xB055, 0xB056, 0xB057, 0xB058, 0xB059, 0x0000, 0x0000, 0xB05A, 0xB05B, 0xB05C, 0xB05D, 0xB05E, 0xB05F, 0xB060, 0xB061 },
	/* 0x007 */ { 0xB062, 0x0000, 0x0000, 0x0000, 0xB063, 0xB064, 0xB065, 0xB066, 0x0000, 0xB067, 0xB068, 0xB069, 0xB06A, 0xB06B, 0xB06C, 0x0000 },
	/* 0x008 */ { 0x0000, 0x0000, 0x0000, 0xB06D, 0xB06E, 0xB06F, 0xB070, 0xB071, 0xB072, 0xB873, 0x0000, 0x0000, 0xB074, 0xB075, 0xB076, 0xB077 },
	/* 0x009 */ { 0xB078, 0xB079, 0x0000, 0x0000, 0xB07A, 0xB07B, 0xB07C, 0xB07D, 0xB07E, 0xB07F, 0xB080, 0xB081, 0xB082, 0xB083, 0xB084, 0xB085 },
	/* 0x00A */ { 0xB086, 0xB087, 0xB088, 0xB089, 0xB08A, 0xB08B, 0x0000, 0x0000, 0xB08C, 0xB08D, 0xB08E, 0xB08F, 0xB090, 0xB091, 0xB092, 0xB093 },
	/* 0x00B */ { 0xB094, 0xB095, 0xB096, 0xB097, 0xB098, 0xB099, 0xB09A, 0xB09B, 0xB09C, 0xB09D, 0xB09E, 0xB09F, 0xB0A0, 0xB0A1, 0xB0A2, 0x0000 },
	/* 0x00C */ { 0xB0A3, 0xB0A4, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB0A5 },
	/* 0x00D */ { 0xB0A6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x00E */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB0A7, 0xB0A8, 0xB0A9 },
	/* 0x00F */ { 0xB0AA, 0xB0AB, 0xB0AC, 0xB0AD, 0xB0AE, 0xC000, 0xC001, 0xC002, 0xC003, 0xC004, 0xC005, 0xC006, 0xC007, 0x0000, 0xC008, 0xC009 },
	/* 0x010 */ { 0xC00A, 0xC00B, 0xB0AF, 0xB0B0, 0x0000, 0x0000, 0xB0B1, 0xB0B2, 0xB0B3, 0xB0B4, 0xB0B5, 0xB0B6, 0xC00C, 0xC00D, 0xB0B7, 0xB0B8 },
	/* 0x011 */ { 0xB0B9, 0x0000, 0x0000, 0x0000, 0xB0BA, 0xB0BB, 0x0000, 0x0000, 0xB0BC, 0xB0BD, 0xC00E, 0xC00F, 0xB0BE, 0xB0BF, 0xB0C0, 0xB0C1 },
	/* 0x012 */ { 0xB0C2, 0xB0C3, 0xB0C4, 0xB0C5, 0xB0C6, 0xB0C7, 0xB0C8, 0xB0C9, 0xB0CA, 0xB0CB, 0xB0CC, 0xB0CD, 0xB0CE, 0xB0CF, 0xB0D0, 0xB0D1 },
	/* 0x013 */ { 0xB0D2, 0xB0D3, 0xB0D4, 0xB0D5, 0xB0D6, 0xB0D7, 0xB0D8, 0xB0D9, 0xB0DA, 0xB0DB, 0xB0DC, 0xB0DD, 0x0000, 0x0000, 0xB0DE, 0xB0DF },
	/* 0x014 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB0E0, 0xB0E1, 0xB0E2, 0xB0E3, 0xC010, 0xC011, 0xC012, 0xC013, 0xB0E4, 0xB0E5 },
	/* 0x015 */ { 0xC014, 0xC015, 0xB0E6, 0xB0E7, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x016 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE8, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADE8, 0xADD8, 0xADDC, 0xADDC, 0xADDC, 0xADDC },
	/* 0x017 */ { 0xADDC, 0xADCA, 0xADCA, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADCA, 0xADCA, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC },
	/* 0x018 */ { 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xAD01, 0xAD01, 0xAD01, 0xAD01, 0xAD01, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x019 */ { 0xD000, 0xD002, 0xADE6, 0xD004, 0xD006, 0xADF0, 0xADE6, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0x0000 },
	/* 0x01A */ { 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE8, 0xADDC, 0xADDC, 0xADE6, 0xADE9, 0xADEA, 0xADEA, 0xADE9 },
	/* 0x01B */ { 0xADEA, 0xADEA, 0xADE9, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x01C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x02B9, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x003B, 0x0000 },
	/* 0x01D */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xB0E8, 0xB0E9, 0x00B7, 0xB0EA, 0xB0EB, 0xB0EC, 0xFFFF, 0xB0ED, 0xFFFF, 0xB0EE, 0xB0EF },
	/* 0x01E */ { 0xC016, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x01F */ { 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB0F0, 0xB0F1, 0xB0F2, 0xB0F3, 0xB0F4, 0xB0F5 },
	/* 0x020 */ { 0xC017, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x021 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB0F6, 0xB0F7, 0xB0F8, 0xB0F9, 0xB0FA, 0x0000 },
	/* 0x022 */ { 0x0000, 0x0000, 0x0000, 0xB0FB, 0xB0FC, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x023 */ { 0xB0FD, 0xB0FE, 0x0000, 0xB0FF, 0x0000, 0x0000, 0x0000, 0xB100, 0x0000, 0x0000, 0x0000, 0x0000, 0xB101, 0xB102, 0xB103, 0x0000 },
	/* 0x024 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB104, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x025 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB105, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x026 */ { 0xB106, 0xB107, 0x0000, 0xB108, 0x0000, 0x0000, 0x0000, 0xB109, 0x0000, 0x0000, 0x0000, 0x0000, 0xB10A, 0xB10B, 0xB10C, 0x0000 },
	/* 0x027 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB10D, 0xB10E, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x028 */ { 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x029 */ { 0x0000, 0xB10F, 0xB110, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x02A */ { 0xB111, 0xB112, 0xB113, 0xB114, 0x0000, 0x0000, 0xB115, 0xB116, 0x0000, 0x0000, 0xB117, 0xB118, 0xB119, 0xB11A, 0xB11B, 0xB11C },
	/* 0x02B */ { 0x0000, 0x0000, 0xB11D, 0xB11E, 0xB11F, 0xB120, 0xB121, 0xB122, 0x0000, 0x0000, 0xB123, 0xB124, 0xB125, 0xB126, 0xB127, 0xB128 },
	/* 0x02C */ { 0xB129, 0xB12A, 0xB12B, 0xB12C, 0xB12D, 0xB12E, 0x0000, 0x0000, 0xB12F, 0xB130, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x02D */ { 0xE800, 0x0000, 0xE801, 0x0000, 0xE802, 0x0000, 0xE803, 0x0000, 0xE804, 0x0000, 0xE805, 0x0000, 0xE806, 0x0000, 0xE807, 0x0000 },
	/* 0x02E */ { 0xE808, 0x0000, 0xE809, 0x0000, 0xE80A, 0x0000, 0xE80B, 0x0000, 0xE80C, 0x0000, 0xE80D, 0x0000, 0xE80E, 0x0000, 0xE80F, 0x0000 },
	/* 0x02F */ { 0xE810, 0x0000, 0xE811, 0x0000, 0xE812, 0x0000, 0xE813, 0x0000, 0xE814, 0x0000, 0xE815, 0x0000, 0xE816, 0x0000, 0xE817, 0x0000 },
	/* 0x030 */ { 0xFFFF, 0xE818, 0xE819, 0xE81A, 0xE81B, 0xE81C, 0xE81D, 0xE81E, 0xE81F, 0xE820, 0xE821, 0xE822, 0xE823, 0xE824, 0xE825, 0xE826 },
	/* 0x031 */ { 0xE827, 0xE828, 0xE829, 0xE82A, 0xE82B, 0xE82C, 0xE82D, 0xE82E, 0xE82F, 0xE830, 0xE831, 0xE832, 0xE833, 0xE834, 0xE835, 0xE836 },
	/* 0x032 */ { 0xE837, 0xE838, 0xE839, 0xE83A, 0xE83B, 0xE83C, 0xE83D, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x033 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB931, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000 },
	/* 0x034 */ { 0xFFFF, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADDE, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x035 */ { 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADDE, 0xADE4, 0xADE6 },
	/* 0x036 */ { 0xAD0A, 0xAD0B, 0xAD0C, 0xAD0D, 0xAD0E, 0xAD0F, 0xAD10, 0xAD11, 0xAD12, 0xAD13, 0xAD13, 0xAD14, 0xAD15, 0xAD16, 0x0000, 0xAD17 },
	/* 0x037 */ { 0x0000, 0xAD18, 0xAD19, 0x0000, 0xADE6, 0xADDC, 0x0000, 0xAD12, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x038 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xAD1E, 0xAD1F, 0xAD20, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x039 */ { 0x0000, 0x0000, 0xB132, 0xB133, 0xB134, 0xB135, 0xB136, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x03A */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD1B, 0xAD1C, 0xAD1D, 0xAD1E, 0xAD1F },
	/* 0x03B */ { 0xAD20, 0xAD21, 0xAD22, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADDC },
	/* 0x03C */ { 0xAD23, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x03D */ { 0xB137, 0x0000, 0xB138, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x03E */ { 0x0000, 0x0000, 0x0000, 0xB139, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000, 0xADE6 },
	/* 0x03F */ { 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0x0000, 0x0000, 0xADE6, 0xADE6, 0x0000, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0x0000, 0x0000 },
	/* 0x040 */ { 0x0000, 0xAD24, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x041 */ { 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADDC, 0xADDC, 0xADE6, 0xADDC, 0xADE6 },
	/* 0x042 */ { 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADDC, 0xADE6, 0xADDC, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000 },
	/* 0x043 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x044 */ { 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xADDC, 0x0000, 0x0000 },
	/* 0x045 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x046 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF },
	/* 0x047 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0xADDC, 0xADDC, 0xFFFF, 0xFFFF, 0x0000, 0xFFFF },
	/* 0x048 */ { 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x049 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADDC },
	/* 0x04A */ { 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x04B */ { 0xADE6, 0xADE6, 0x0000, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADDC },
	/* 0x04C */ { 0xAD1B, 0xAD1C, 0xAD1D, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x04D */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB13A, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x04E */ { 0x0000, 0xB13B, 0x0000, 0x0000, 0xB13C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0x0000, 0x0000 },
	/* 0x04F */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0x0000, 0x0000 },
	/* 0x050 */ { 0x0000, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0xB13D, 0xB13E, 0xB13F, 0xB140, 0xB141, 0xB142, 0xB143, 0xB144 },
	/* 0x051 */ { 0x0000, 0xFFFF, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xAD07, 0x0000, 0x0000, 0x0000 },
	/* 0x052 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xB145, 0xB146, 0xAD09, 0x0000, 0xFFFF },
	/* 0x053 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xB147, 0xB148, 0xFFFF, 0xB149 },
	/* 0x054 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xFFFF },
	/* 0x055 */ { 0x0000, 0xFFFF, 0x0000, 0xB14A, 0xFFFF, 0x0000, 0xB14B, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xAD07, 0xFFFF, 0x0000, 0x0000 },
	/* 0x056 */ { 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF },
	/* 0x057 */ { 0xFFFF, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xB14C, 0xB14D, 0xB14E, 0x0000, 0xFFFF, 0xB14F, 0xFFFF },
	/* 0x058 */ { 0x0000, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xAD07, 0x0000, 0x0000, 0x0000 },
	/* 0x059 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF },
	/* 0x05A */ { 0x0000, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xAD07, 0x0000, 0x0000, 0x0000 },
	/* 0x05B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0xB150, 0xFFFF, 0xFFFF, 0xB151, 0xB152, 0xAD09, 0xFFFF, 0xFFFF },
	/* 0x05C */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xB153, 0xB154, 0xFFFF, 0x0000 },
	/* 0x05D */ { 0x0000, 0xFFFF, 0x0000, 0x0000, 0xB155, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0x0000, 0xFFFF, 0x0000, 0x0000 },
	/* 0x05E */ { 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xB156, 0xB157, 0xB158, 0xAD09, 0xFFFF, 0xFFFF },
	/* 0x05F */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xAD07, 0x0000, 0x0000, 0x0000 },
	/* 0x060 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0xB159, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF },
	/* 0x061 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAD54, 0xAD5B, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0xFFFF, 0xFFFF },
	/* 0x062 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xAD07, 0x0000, 0x0000, 0x0000 },
	/* 0x063 */ { 0xB15A, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0xB15B, 0xB15C, 0xFFFF, 0xB15D, 0xC018, 0x0000, 0xAD09, 0xFFFF, 0xFFFF },
	/* 0x064 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xAD09, 0x0000, 0x0000, 0x0000 },
	/* 0x065 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xB15E, 0xB15F, 0xB160, 0xAD09, 0x0000, 0x0000 },
	/* 0x066 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000 },
	/* 0x067 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0xFFFF, 0x0000, 0x0000, 0xB161, 0x0000, 0xB162, 0xC019, 0xB163, 0x0000 },
	/* 0x068 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD67, 0xAD67, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000 },
	/* 0x069 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD6B, 0xAD6B, 0xAD6B, 0xAD6B, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x06A */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD76, 0xAD76, 0xAD09, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF },
	/* 0x06B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0xFFFF, 0xAD7A, 0xAD7A, 0xAD7A, 0xAD7A, 0x0000, 0x0000, 0x0000, 0xFFFF },
	/* 0x06C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0xADDC, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x06D */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0x0000, 0xADDC, 0x0000, 0xADD8, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x06E */ { 0x0000, 0x0000, 0x0000, 0xB164, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0xB165, 0x0000, 0x0000 },
	/* 0x06F */ { 0x0000, 0x0000, 0xB166, 0x0000, 0x0000, 0x0000, 0x0000, 0xB167, 0x0000, 0x0000, 0x0000, 0x0000, 0xB168, 0x0000, 0x0000, 0x0000 },
	/* 0x070 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB169, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x071 */ { 0xFFFF, 0xAD81, 0xAD82, 0xD009, 0xAD84, 0xD00C, 0xB16A, 0x0000, 0xB16B, 0x0000, 0xAD82, 0xAD82, 0xAD82, 0xAD82, 0x0000, 0x0000 },
	/* 0x072 */ { 0xAD82, 0xD00F, 0xADE6, 0xADE6, 0xAD09, 0x0000, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x073 */ { 0x0000, 0x0000, 0x0000, 0xB16C, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0xB16D, 0x0000, 0x0000 },
	/* 0x074 */ { 0x0000, 0x0000, 0xB16E, 0x0000, 0x0000, 0x0000, 0x0000, 0xB16F, 0x0000, 0x0000, 0x0000, 0x0000, 0xB170, 0x0000, 0x0000, 0x0000 },
	/* 0x075 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB171, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000 },
	/* 0x076 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000 },
	/* 0x077 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB172, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x078 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0xAD09, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x079 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0x0000, 0x0000 },
	/* 0x07A */ { 0xE83E, 0xE83F, 0xE840, 0xE841, 0xE842, 0xE843, 0xE844, 0xE845, 0xE846, 0xE847, 0xE848, 0xE849, 0xE84A, 0xE84B, 0xE84C, 0xE84D },
	/* 0x07B */ { 0xE84E, 0xE84F, 0xE850, 0xE851, 0xE852, 0xE853, 0xE854, 0xE855, 0xE856, 0xE857, 0xE858, 0xE859, 0xE85A, 0xE85B, 0xE85C, 0xE85D },
	/* 0x07C */ { 0xE85E, 0xE85F, 0xE860, 0xE861, 0xE862, 0xE863, 0xFFFF, 0xE864, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xE865, 0xFFFF, 0xFFFF },
	/* 0x07D */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x07E */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xE866, 0xE867, 0xE868, 0xE869, 0xE86A, 0xE86B, 0xFFFF, 0xFFFF },
	/* 0x07F */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000 },
	/* 0x080 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x081 */ { 0x0000, 0x0000, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xFFFF, 0xFFFF },
	/* 0x082 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE4, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x083 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDE, 0xADE6, 0xADDC, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x084 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADDC, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000 },
	/* 0x085 */ { 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x086 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF, 0xADDC },
	/* 0x087 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0x0000, 0xADDC },
	/* 0x088 */ { 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF },
	/* 0x089 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB173, 0x0000, 0xB174, 0x0000, 0xB175, 0x0000, 0xB176, 0x0000, 0xB177, 0x0000 },
	/* 0x08A */ { 0x0000, 0x0000, 0xB178, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x08B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB179, 0x0000, 0xB17A, 0x0000, 0x0000 },
	/* 0x08C */ { 0xB17B, 0xB17C, 0x0000, 0xB17D, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000 },
	/* 0x08D */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x08E */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x08F */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x090 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x091 */ { 0x0000, 0x0000, 0xAD09, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x092 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD07, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x093 */ { 0xE86C, 0xE86D, 0xE86E, 0xE86F, 0xE870, 0xE871, 0xE872, 0xE873, 0xE874, 0xE875, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x094 */ { 0xE876, 0xE877, 0xE878, 0xE879, 0xE87A, 0xE87B, 0xE87C, 0xE87D, 0xE87E, 0xE87F, 0xE880, 0xE881, 0xE882, 0xE883, 0xE884, 0xE885 },
	/* 0x095 */ { 0xE886, 0xE887, 0xE888, 0xE889, 0xE88A, 0xE88B, 0xE88C, 0xE88D, 0xE88E, 0xE88F, 0xE890, 0xE891, 0xE892, 0xE893, 0xE894, 0xE895 },
	/* 0x096 */ { 0xE896, 0xE897, 0xE898, 0xE899, 0xE89A, 0xE89B, 0xE89C, 0xE89D, 0xE89E, 0xE89F, 0xE8A0, 0xFFFF, 0xFFFF, 0xE8A1, 0xE8A2, 0xE8A3 },
	/* 0x097 */ { 0xADE6, 0xADE6, 0xADE6, 0x0000, 0xAD01, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADDC, 0xADDC },
	/* 0x098 */ { 0xADE6, 0x0000, 0xAD01, 0xAD01, 0xAD01, 0xAD01, 0xAD01, 0xAD01, 0xAD01, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0x0000, 0x0000 },
	/* 0x099 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x09A */ { 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADE6, 0xADEA, 0xADD6, 0xADDC },
	/* 0x09B */ { 0xADCA, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x09C */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE8, 0xADE4, 0xADE4, 0xADDC, 0xADDA, 0xADE6, 0xADE9, 0xADDC, 0xADE6, 0xADDC },
	/* 0x09D */ { 0xB17E, 0xB17F, 0xB180, 0xB181, 0xB182, 0xB183, 0xB184, 0xB185, 0xC01A, 0xC01B, 0xB186, 0xB187, 0xB188, 0xB189, 0xB18A, 0xB18B },
	/* 0x09E */ { 0xB18C, 0xB18D, 0xB18E, 0xB18F, 0xC01C, 0xC01D, 0xC01E, 0xC01F, 0xB190, 0xB191, 0xB192, 0xB193, 0xC020, 0xC021, 0xB194, 0xB195 },
	/* 0x09F */ { 0xB196, 0xB197, 0xB198, 0xB199, 0xB19A, 0xB19B, 0xB19C, 0xB19D, 0xB19E, 0xB19F, 0xB1A0, 0xB1A1, 0xB1A2, 0xB1A3, 0xC022, 0xC023 },
	/* 0x0A0 */ { 0xB1A4, 0xB1A5, 0xB1A6, 0xB1A7, 0xB1A8, 0xB1A9, 0xB1AA, 0xB1AB, 0xC024, 0xC025, 0xB1AC, 0xB1AD, 0xB1AE, 0xB1AF, 0xB1B0, 0xB1B1 },
	/* 0x0A1 */ { 0xB1B2, 0xB1B3, 0xB1B4, 0xB1B5, 0xB1B6, 0xB1B7, 0xB1B8, 0xB1B9, 0xB1BA, 0xB1BB, 0xB1BC, 0xB1BD, 0xC026, 0xC027, 0xC028, 0xC029 },
	/* 0x0A2 */ { 0xC02A, 0xC02B, 0xC02C, 0xC02D, 0xB1BE, 0xB1BF, 0xB1C0, 0xB1C1, 0xB1C2, 0xB1C3, 0xB1C4, 0xB1C5, 0xC02E, 0xC02F, 0xB1C6, 0xB1C7 },
	/* 0x0A3 */ { 0xB1C8, 0xB1C9, 0xB1CA, 0xB1CB, 0xC030, 0xC031, 0xC032, 0xC033, 0xC034, 0xC035, 0xB1CC, 0xB1CD, 0xB1CE, 0xB1CF, 0xB1D0, 0xB1D1 },
	/* 0x0A4 */ { 0xB1D2, 0xB1D3, 0xB1D4, 0xB1D5, 0xB1D6, 0xB1D7, 0xB1D8, 0xB1D9, 0xC036, 0xC037, 0xC038, 0xC039, 0xB1DA, 0xB1DB, 0xB1DC, 0xB1DD },
	/* 0x0A5 */ { 0xB1DE, 0xB1DF, 0xB1E0, 0xB1E1, 0xB1E2, 0xB1E3, 0xB1E4, 0xB1E5, 0xB1E6, 0xB1E7, 0xB1E8, 0xB1E9, 0xB1EA, 0xB1EB, 0xB1EC, 0xB1ED },
	/* 0x0A6 */ { 0xB1EE, 0xB1EF, 0xB1F0, 0xB1F1, 0xB1F2, 0xB1F3, 0xB1F4, 0xB1F5, 0xB1F6, 0xB1F7, 0xB9F8, 0xB1F9, 0x0000, 0x0000, 0xB9FA, 0x0000 },
	/* 0x0A7 */ { 0xB1FB, 0xB1FC, 0xB1FD, 0xB1FE, 0xC03A, 0xC03B, 0xC03C, 0xC03D, 0xC03E, 0xC03F, 0xC040, 0xC041, 0xC042, 0xC043, 0xC044, 0xC045 },
	/* 0x0A8 */ { 0xC046, 0xC047, 0xC048, 0xC049, 0xC04A, 0xC04B, 0xC04C, 0xC04D, 0xB1FF, 0xB200, 0xB201, 0xB202, 0xB203, 0xB204, 0xC04E, 0xC04F },
	/* 0x0A9 */ { 0xC050, 0xC051, 0xC052, 0xC053, 0xC054, 0xC055, 0xC056, 0xC057, 0xB205, 0xB206, 0xB207, 0xB208, 0xB209, 0xB20A, 0xB20B, 0xB20C },
	/* 0x0AA */ { 0xC058, 0xC059, 0xC05A, 0xC05B, 0xC05C, 0xC05D, 0xC05E, 0xC05F, 0xC060, 0xC061, 0xC062, 0xC063, 0xC064, 0xC065, 0xC066, 0xC067 },
	/* 0x0AB */ { 0xC068, 0xC069, 0xC06A, 0xC06B, 0xB20D, 0xB20E, 0xB20F, 0xB210, 0xC06C, 0xC06D, 0xC06E, 0xC06F, 0xC070, 0xC071, 0xC072, 0xC073 },
	/* 0x0AC */ { 0xC074, 0xC075, 0xB211, 0xB212, 0xB213, 0xB214, 0xB215, 0xB216, 0xB217, 0xB218, 0xE8A4, 0x0000, 0xE8A5, 0x0000, 0xE8A6, 0x0000 },
	/* 0x0AD */ { 0xB219, 0xB21A, 0xC076, 0xC077, 0xC078, 0xC079, 0xC07A, 0xC07B, 0xB21B, 0xB21C, 0xC07C, 0xC07D, 0xC07E, 0xC07F, 0xC080, 0xC081 },
	/* 0x0AE */ { 0xB21D, 0xB21E, 0xC082, 0xC083, 0xC084, 0xC085, 0xFFFF, 0xFFFF, 0xB21F, 0xB220, 0xC086, 0xC087, 0xC088, 0xC089, 0xFFFF, 0xFFFF },
	/* 0x0AF */ { 0xB221, 0xB222, 0xC08A, 0xC08B, 0xC08C, 0xC08D, 0xC08E, 0xC08F, 0xB223, 0xB224, 0xC090, 0xC091, 0xC092, 0xC093, 0xC094, 0xC095 },
	/* 0x0B0 */ { 0xB225, 0xB226, 0xC096, 0xC097, 0xC098, 0xC099, 0xC09A, 0xC09B, 0xB227, 0xB228, 0xC09C, 0xC09D, 0xC09E, 0xC09F, 0xC0A0, 0xC0A1 },
	/* 0x0B1 */ { 0xB229, 0xB22A, 0xC0A2, 0xC0A3, 0xC0A4, 0xC0A5, 0xFFFF, 0xFFFF, 0xB22B, 0xB22C, 0xC0A6, 0xC0A7, 0xC0A8, 0xC0A9, 0xFFFF, 0xFFFF },
	/* 0x0B2 */ { 0xB22D, 0xB22E, 0xC0AA, 0xC0AB, 0xC0AC, 0xC0AD, 0xC0AE, 0xC0AF, 0xFFFF, 0xB22F, 0xFFFF, 0xC0B0, 0xFFFF, 0xC0B1, 0xFFFF, 0xC0B2 },
	/* 0x0B3 */ { 0xB230, 0xB231, 0xC0B3, 0xC0B4, 0xC0B5, 0xC0B6, 0xC0B7, 0xC0B8, 0xB232, 0xB233, 0xC0B9, 0xC0BA, 0xC0BB, 0xC0BC, 0xC0BD, 0xC0BE },
	/* 0x0B4 */ { 0xB234, 0xB235, 0xB236, 0xB237, 0xB238, 0xB239, 0xB23A, 0xB23B, 0xB23C, 0xB23D, 0xB23E, 0xB23F, 0xB240, 0xB241, 0xFFFF, 0xFFFF },
	/* 0x0B5 */ { 0xC0BF, 0xC0C0, 0xD012, 0xD017, 0xD01C, 0xD021, 0xD026, 0xD02B, 0xC0C1, 0xC0C2, 0xD030, 0xD035, 0xD03A, 0xD03F, 0xD044, 0xD049 },
	/* 0x0B6 */ { 0xC0C3, 0xC0C4, 0xD04E, 0xD053, 0xD058, 0xD05D, 0xD062, 0xD067, 0xC0C5, 0xC0C6, 0xD06C, 0xD071, 0xD076, 0xD07B, 0xD080, 0xD085 },
	/* 0x0B7 */ { 0xC0C7, 0xC0C8, 0xD08A, 0xD08F, 0xD094, 0xD099, 0xD09E, 0xD0A3, 0xC0C9, 0xC0CA, 0xD0A8, 0xD0AD, 0xD0B2, 0xD0B7, 0xD0BC, 0xD0C1 },
	/* 0x0B8 */ { 0xB242, 0xB243, 0xC0CB, 0xB244, 0xC0CC, 0xFFFF, 0xB245, 0xC0CD, 0xB246, 0xB247, 0xB248, 0xB249, 0xB24A, 0x0000, 0x03B9, 0x0000 },
	/* 0x0B9 */ { 0x0000, 0xB24B, 0xC0CE, 0xB24C, 0xC0CF, 0xFFFF, 0xB24D, 0xC0D0, 0xB24E, 0xB24F, 0xB250, 0xB251, 0xB252, 0xB253, 0xB254, 0xB255 },
	/* 0x0BA */ { 0xB256, 0xB257, 0xC0D1, 0xC0D2, 0xFFFF, 0xFFFF, 0xB258, 0xC0D3, 0xB259, 0xB25A, 0xB25B, 0xB25C, 0xFFFF, 0xB25D, 0xB25E, 0xB25F },
	/* 0x0BB */ { 0xB260, 0xB261, 0xC0D4, 0xC0D5, 0xB262, 0xB263, 0xB264, 0xC0D6, 0xB265, 0xB266, 0xB267, 0xB268, 0xB269, 0xB26A, 0xB26B, 0x0060 },
	/* 0x0BC */ { 0xFFFF, 0xFFFF, 0xC0D7, 0xB26C, 0xC0D8, 0xFFFF, 0xB26D, 0xC0D9, 0xB26E, 0xB26F, 0xB270, 0xB271, 0xB272, 0x00B4, 0x0000, 0xFFFF },
	/* 0x0BD */ { 0x2002, 0x2003, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0BE */ { 0xADE6, 0xADE6, 0xAD01, 0xAD01, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xAD01, 0xAD01, 0xAD01, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000 },
	/* 0x0BF */ { 0x0000, 0xADE6, 0x0000, 0x0000, 0x0000, 0xAD01, 0xAD01, 0xADE6, 0xADDC, 0xADE6, 0xAD01, 0xAD01, 0xADDC, 0xADDC, 0xADDC, 0xADDC },
	/* 0x0C0 */ { 0xADE6, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x0C1 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x03A9, 0x0000, 0x0000, 0x0000, 0x004B, 0xB273, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0C2 */ { 0x0000, 0x0000, 0xE8A7, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0C3 */ { 0xE8A8, 0xE8A9, 0xE8AA, 0xE8AB, 0xE8AC, 0xE8AD, 0xE8AE, 0xE8AF, 0xE8B0, 0xE8B1, 0xE8B2, 0xE8B3, 0xE8B4, 0xE8B5, 0xE8B6, 0xE8B7 },
	/* 0x0C4 */ { 0x0000, 0x0000, 0x0000, 0xE8B8, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x0C5 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB274, 0xB275, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0C6 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB276, 0x0000 },
	/* 0x0C7 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB277, 0xB278, 0xB279 },
	/* 0x0C8 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xB27A, 0x0000, 0x0000, 0x0000, 0x0000, 0xB27B, 0x0000, 0x0000, 0xB27C, 0x0000, 0x0000, 0x0000 },
	/* 0x0C9 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xB27D, 0x0000, 0xB27E, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0CA */ { 0x0000, 0xB27F, 0x0000, 0x0000, 0xB280, 0x0000, 0x0000, 0xB281, 0x0000, 0xB282, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0CB */ { 0xB283, 0x0000, 0xB284, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB285, 0xB286, 0xB287 },
	/* 0x0CC */ { 0xB288, 0xB289, 0x0000, 0x0000, 0xB28A, 0xB28B, 0x0000, 0x0000, 0xB28C, 0xB28D, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0CD */ { 0xB28E, 0xB28F, 0x0000, 0x0000, 0xB290, 0xB291, 0x0000, 0x0000, 0xB292, 0xB293, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0CE */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB294, 0xB295, 0xB296, 0xB297 },
	/* 0x0CF */ { 0xB298, 0xB299, 0xB29A, 0xB29B, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB29C, 0xB29D, 0xB29E, 0xB29F, 0x0000, 0x0000 },
	/* 0x0D0 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x3008, 0x3009, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0D1 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xE8B9, 0xE8BA, 0xE8BB, 0xE8BC, 0xE8BD, 0xE8BE, 0xE8BF, 0xE8C0, 0xE8C1, 0xE8C2 },
	/* 0x0D2 */ { 0xE8C3, 0xE8C4, 0xE8C5, 0xE8C6, 0xE8C7, 0xE8C8, 0xE8C9, 0xE8CA, 0xE8CB, 0xE8CC, 0xE8CD, 0xE8CE, 0xE8CF, 0xE8D0, 0xE8D1, 0xE8D2 },
	/* 0x0D3 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB2A0, 0x0000, 0x0000, 0x0000 },
	/* 0x0D4 */ { 0xE8D3, 0xE8D4, 0xE8D5, 0xE8D6, 0xE8D7, 0xE8D8, 0xE8D9, 0xE8DA, 0xE8DB, 0xE8DC, 0xE8DD, 0xE8DE, 0xE8DF, 0xE8E0, 0xE8E1, 0xE8E2 },
	/* 0x0D5 */ { 0xE8E3, 0xE8E4, 0xE8E5, 0xE8E6, 0xE8E7, 0xE8E8, 0xE8E9, 0xE8EA, 0xE8EB, 0xE8EC, 0xE8ED, 0xE8EE, 0xE8EF, 0xE8F0, 0xE8F1, 0xE8F2 },
	/* 0x0D6 */ { 0xE8F3, 0xE8F4, 0xE8F5, 0xE8F6, 0xE8F7, 0xE8F8, 0xE8F9, 0xE8FA, 0xE8FB, 0xE8FC, 0xE8FD, 0xE8FE, 0xE8FF, 0xE900, 0xE901, 0xE902 },
	/* 0x0D7 */ { 0xE903, 0x0000, 0xE904, 0xE905, 0xE906, 0x0000, 0x0000, 0xE907, 0x0000, 0xE908, 0x0000, 0xE909, 0x0000, 0xE90A, 0xE90B, 0xE90C },
	/* 0x0D8 */ { 0xE90D, 0x0000, 0xE90E, 0x0000, 0x0000, 0xE90F, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xE910, 0xE911 },
	/* 0x0D9 */ { 0xE912, 0x0000, 0xE913, 0x0000, 0xE914, 0x0000, 0xE915, 0x0000, 0xE916, 0x0000, 0xE917, 0x0000, 0xE918, 0x0000, 0xE919, 0x0000 },
	/* 0x0DA */ { 0xE91A, 0x0000, 0xE91B, 0x0000, 0xE91C, 0x0000, 0xE91D, 0x0000, 0xE91E, 0x0000, 0xE91F, 0x0000, 0xE920, 0x0000, 0xE921, 0x0000 },
	/* 0x0DB */ { 0xE922, 0x0000, 0xE923, 0x0000, 0xE924, 0x0000, 0xE925, 0x0000, 0xE926, 0x0000, 0xE927, 0x0000, 0xE928, 0x0000, 0xE929, 0x0000 },
	/* 0x0DC */ { 0xE92A, 0x0000, 0xE92B, 0x0000, 0xE92C, 0x0000, 0xE92D, 0x0000, 0xE92E, 0x0000, 0xE92F, 0x0000, 0xE930, 0x0000, 0xE931, 0x0000 },
	/* 0x0DD */ { 0xE932, 0x0000, 0xE933, 0x0000, 0xE934, 0x0000, 0xE935, 0x0000, 0xE936, 0x0000, 0xE937, 0x0000, 0xE938, 0x0000, 0xE939, 0x0000 },
	/* 0x0DE */ { 0xE93A, 0x0000, 0xE93B, 0x0000, 0xE93C, 0x0000, 0xE93D, 0x0000, 0xE93E, 0x0000, 0xE93F, 0x0000, 0xE940, 0x0000, 0xE941, 0x0000 },
	/* 0x0DF */ { 0xE942, 0x0000, 0xE943, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xE944, 0x0000, 0xE945, 0x0000, 0xADE6 },
	/* 0x0E0 */ { 0xADE6, 0xADE6, 0xE946, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0E1 */ { 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAD09 },
	/* 0x0E2 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDA, 0xADE4, 0xADE8, 0xADDE, 0xADE0, 0xADE0 },
	/* 0x0E3 */ { 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB2A1, 0x0000, 0xB2A2, 0x0000 },
	/* 0x0E4 */ { 0xB2A3, 0x0000, 0xB2A4, 0x0000, 0xB2A5, 0x0000, 0xB2A6, 0x0000, 0xB2A7, 0x0000, 0xB2A8, 0x0000, 0xB2A9, 0x0000, 0xB2AA, 0x0000 },
	/* 0x0E5 */ { 0xB2AB, 0x0000, 0xB2AC, 0x0000, 0x0000, 0xB2AD, 0x0000, 0xB2AE, 0x0000, 0xB2AF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0E6 */ { 0xB2B0, 0xB2B1, 0x0000, 0xB2B2, 0xB2B3, 0x0000, 0xB2B4, 0xB2B5, 0x0000, 0xB2B6, 0xB2B7, 0x0000, 0xB2B8, 0xB2B9, 0x0000, 0x0000 },
	/* 0x0E7 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xB2BA, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xAD08, 0xAD08, 0x0000, 0x0000, 0x0000, 0xB2BB, 0x0000 },
	/* 0x0E8 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB2BC, 0x0000, 0xB2BD, 0x0000 },
	/* 0x0E9 */ { 0xB2BE, 0x0000, 0xB2BF, 0x0000, 0xB2C0, 0x0000, 0xB2C1, 0x0000, 0xB2C2, 0x0000, 0xB2C3, 0x0000, 0xB2C4, 0x0000, 0xB2C5, 0x0000 },
	/* 0x0EA */ { 0xB2C6, 0x0000, 0xB2C7, 0x0000, 0x0000, 0xB2C8, 0x0000, 0xB2C9, 0x0000, 0xB2CA, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x0EB */ { 0xB2CB, 0xB2CC, 0x0000, 0xB2CD, 0xB2CE, 0x0000, 0xB2CF, 0xB2D0, 0x0000, 0xB2D1, 0xB2D2, 0x0000, 0xB2D3, 0xB2D4, 0x0000, 0x0000 },
	/* 0x0EC */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xB2D5, 0x0000, 0x0000, 0xB2D6, 0xB2D7, 0xB2D8, 0xB2D9, 0x0000, 0x0000, 0x0000, 0xB2DA, 0x0000 },
	/* 0x0ED */ { 0xE947, 0x0000, 0xE948, 0x0000, 0xE949, 0x0000, 0xE94A, 0x0000, 0xE94B, 0x0000, 0xE94C, 0x0000, 0xE94D, 0x0000, 0xE94E, 0x0000 },
	/* 0x0EE */ { 0xE94F, 0x0000, 0xE950, 0x0000, 0xE951, 0x0000, 0xE952, 0x0000, 0xE953, 0x0000, 0xE954, 0x0000, 0xE955, 0x0000, 0xE956, 0x0000 },
	/* 0x0EF */ { 0xE957, 0x0000, 0xE958, 0x0000, 0xE959, 0x0000, 0xE95A, 0x0000, 0xE95B, 0x0000, 0xE95C, 0x0000, 0xE95D, 0x0000, 0x0000, 0xADE6 },
	/* 0x0F0 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000 },
	/* 0x0F1 */ { 0xE95E, 0x0000, 0xE95F, 0x0000, 0xE960, 0x0000, 0xE961, 0x0000, 0xE962, 0x0000, 0xE963, 0x0000, 0xE964, 0x0000, 0xE965, 0x0000 },
	/* 0x0F2 */ { 0xE966, 0x0000, 0xE967, 0x0000, 0xE968, 0x0000, 0xE969, 0x0000, 0xE96A, 0x0000, 0xE96B, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6 },
	/* 0x0F3 */ { 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x0F4 */ { 0x0000, 0x0000, 0xE96C, 0x0000, 0xE96D, 0x0000, 0xE96E, 0x0000, 0xE96F, 0x0000, 0xE970, 0x0000, 0xE971, 0x0000, 0xE972, 0x0000 },
	/* 0x0F5 */ { 0x0000, 0x0000, 0xE973, 0x0000, 0xE974, 0x0000, 0xE975, 0x0000, 0xE976, 0x0000, 0xE977, 0x0000, 0xE978, 0x0000, 0xE979, 0x0000 },
	/* 0x0F6 */ { 0xE97A, 0x0000, 0xE97B, 0x0000, 0xE97C, 0x0000, 0xE97D, 0x0000, 0xE97E, 0x0000, 0xE97F, 0x0000, 0xE980, 0x0000, 0xE981, 0x0000 },
	/* 0x0F7 */ { 0xE982, 0x0000, 0xE983, 0x0000, 0xE984, 0x0000, 0xE985, 0x0000, 0xE986, 0x0000, 0xE987, 0x0000, 0xE988, 0x0000, 0xE989, 0x0000 },
	/* 0x0F8 */ { 0xE98A, 0x0000, 0xE98B, 0x0000, 0xE98C, 0x0000, 0xE98D, 0x0000, 0xE98E, 0x0000, 0xE98F, 0x0000, 0xE990, 0x0000, 0xE991, 0x0000 },
	/* 0x0F9 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xE992, 0x0000, 0xE993, 0x0000, 0xE994, 0xE995, 0x0000 },
	/* 0x0FA */ { 0xE996, 0x0000, 0xE997, 0x0000, 0xE998, 0x0000, 0xE999, 0x0000, 0x0000, 0x0000, 0x0000, 0xE99A, 0x0000, 0xE99B, 0x0000, 0x0000 },
	/* 0x0FB */ { 0xE99C, 0x0000, 0xE99D, 0x0000, 0x0000, 0x0000, 0xE99E, 0x0000, 0xE99F, 0x0000, 0xE9A0, 0x0000, 0xE9A1, 0x0000, 0xE9A2, 0x0000 },
	/* 0x0FC */ { 0xE9A3, 0x0000, 0xE9A4, 0x0000, 0xE9A5, 0x0000, 0xE9A6, 0x0000, 0xE9A7, 0x0000, 0xE9A8, 0xE9A9, 0xE9AA, 0xE9AB, 0xE9AC, 0x0000 },
	/* 0x0FD */ { 0xE9AD, 0xE9AE, 0xE9AF, 0xE9B0, 0xE9B1, 0x0000, 0xE9B2, 0x0000, 0xE9B3, 0x0000, 0xE9B4, 0x0000, 0xE9B5, 0x0000, 0xE9B6, 0x0000 },
	/* 0x0FE */ { 0xE9B7, 0x0000, 0xE9B8, 0x0000, 0xE9B9, 0xE9BA, 0xE9BB, 0xE9BC, 0x0000, 0xE9BD, 0x0000, 0xE9BE, 0xE9BF, 0x0000, 0xFFFF, 0xFFFF },
	/* 0x0FF */ { 0xE9C0, 0x0000, 0xFFFF, 0x0000, 0xFFFF, 0x0000, 0xE9C1, 0x0000, 0xE9C2, 0x0000, 0xE9C3, 0x0000, 0xE9C4, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x100 */ { 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0xE9C5, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x101 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x102 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x103 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000 },
	/* 0x104 */ { 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x105 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0xADDC, 0xADDC, 0x0000, 0x0000 },
	/* 0x106 */ { 0x0000, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000 },
	/* 0x107 */ { 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x108 */ { 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000 },
	/* 0x109 */ { 0xADE6, 0x0000, 0xADE6, 0xADE6, 0xADDC, 0x0000, 0x0000, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6 },
	/* 0x10A */ { 0x0000, 0xADE6, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x10B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x10C */ { 0xE9C6, 0xE9C7, 0xE9C8, 0xE9C9, 0xE9CA, 0xE9CB, 0xE9CC, 0xE9CD, 0xE9CE, 0xE9CF, 0xE9D0, 0xE9D1, 0xE9D2, 0xE9D3, 0xE9D4, 0xE9D5 },
	/* 0x10D */ { 0xE9D6, 0xE9D7, 0xE9D8, 0xE9D9, 0xE9DA, 0xE9DB, 0xE9DC, 0xE9DD, 0xE9DE, 0xE9DF, 0xE9E0, 0xE9E1, 0xE9E2, 0xE9E3, 0xE9E4, 0xE9E5 },
	/* 0x10E */ { 0xE9E6, 0xE9E7, 0xE9E8, 0xE9E9, 0xE9EA, 0xE9EB, 0xE9EC, 0xE9ED, 0xE9EE, 0xE9EF, 0xE9F0, 0xE9F1, 0xE9F2, 0xE9F3, 0xE9F4, 0xE9F5 },
	/* 0x10F */ { 0xE9F6, 0xE9F7, 0xE9F8, 0xE9F9, 0xE9FA, 0xE9FB, 0xE9FC, 0xE9FD, 0xE9FE, 0xE9FF, 0xEA00, 0xEA01, 0xEA02, 0xEA03, 0xEA04, 0xEA05 },
	/* 0x110 */ { 0xEA06, 0xEA07, 0xEA08, 0xEA09, 0xEA0A, 0xEA0B, 0xEA0C, 0xEA0D, 0xEA0E, 0xEA0F, 0xEA10, 0xEA11, 0xEA12, 0xEA13, 0xEA14, 0xEA15 },
	/* 0x111 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF },
	/* 0x112 */ { 0xAC00, 0xAC00, 0xAC00, 0xAC00, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x113 */ { 0x8C48, 0x66F4, 0x8ECA, 0x8CC8, 0x6ED1, 0x4E32, 0x53E5, 0x9F9C, 0x9F9C, 0x5951, 0x91D1, 0x5587, 0x5948, 0x61F6, 0x7669, 0x7F85 },
	/* 0x114 */ { 0x863F, 0x87BA, 0x88F8, 0x908F, 0x6A02, 0x6D1B, 0x70D9, 0x73DE, 0x843D, 0x916A, 0x99F1, 0x4E82, 0x5375, 0x6B04, 0x721B, 0x862D },
	/* 0x115 */ { 0x9E1E, 0x5D50, 0x6FEB, 0x85CD, 0x8964, 0x62C9, 0x81D8, 0x881F, 0x5ECA, 0x6717, 0x6D6A, 0x72FC, 0x90CE, 0x4F86, 0x51B7, 0x52DE },
	/* 0x116 */ { 0x64C4, 0x6AD3, 0x7210, 0x76E7, 0x8001, 0x8606, 0x865C, 0x8DEF, 0x9732, 0x9B6F, 0x9DFA, 0x788C, 0x797F, 0x7DA0, 0x83C9, 0x9304 },
	/* 0x117 */ { 0x9E7F, 0x8AD6, 0x58DF, 0x5F04, 0x7C60, 0x807E, 0x7262, 0x78CA, 0x8CC2, 0x96F7, 0x58D8, 0x5C62, 0x6A13, 0x6DDA, 0x6F0F, 0x7D2F },
	/* 0x118 */ { 0x7E37, 0x964B, 0x52D2, 0x808B, 0x51DC, 0x51CC, 0x7A1C, 0x7DBE, 0x83F1, 0x9675, 0x8B80, 0x62CF, 0x6A02, 0x8AFE, 0x4E39, 0x5BE7 },
	/* 0x119 */ { 0x6012, 0x7387, 0x7570, 0x5317, 0x78FB, 0x4FBF, 0x5FA9, 0x4E0D, 0x6CCC, 0x6578, 0x7D22, 0x53C3, 0x585E, 0x7701, 0x8449, 0x8AAA },
	/* 0x11A */ { 0x6BBA, 0x8FB0, 0x6C88, 0x62FE, 0x82E5, 0x63A0, 0x7565, 0x4EAE, 0x5169, 0x51C9, 0x6881, 0x7CE7, 0x826F, 0x8AD2, 0x91CF, 0x52F5 },
	/* 0x11B */ { 0x5442, 0x5973, 0x5EEC, 0x65C5, 0x6FFE, 0x792A, 0x95AD, 0x9A6A, 0x9E97, 0x9ECE, 0x529B, 0x66C6, 0x6B77, 0x8F62, 0x5E74, 0x6190 },
	/* 0x11C */ { 0x6200, 0x649A, 0x6F23, 0x7149, 0x7489, 0x79CA, 0x7DF4, 0x806F, 0x8F26, 0x84EE, 0x9023, 0x934A, 0x5217, 0x52A3, 0x54BD, 0x70C8 },
	/* 0x11D */ { 0x88C2, 0x8AAA, 0x5EC9, 0x5FF5, 0x637B, 0x6BAE, 0x7C3E, 0x7375, 0x4EE4, 0x56F9, 0x5BE7, 0x5DBA, 0x601C, 0x73B2, 0x7469, 0x7F9A },
	/* 0x11E */ { 0x8046, 0x9234, 0x96F6, 0x9748, 0x9818, 0x4F8B, 0x79AE, 0x91B4, 0x96B8, 0x60E1, 0x4E86, 0x50DA, 0x5BEE, 0x5C3F, 0x6599, 0x6A02 },
	/* 0x11F */ { 0x71CE, 0x7642, 0x84FC, 0x907C, 0x9F8D, 0x6688, 0x962E, 0x5289, 0x677B, 0x67F3, 0x6D41, 0x6E9C, 0x7409, 0x7559, 0x786B, 0x7D10 },
	/* 0x120 */ { 0x985E, 0x516D, 0x622E, 0x9678, 0x502B, 0x5D19, 0x6DEA, 0x8F2A, 0x5F8B, 0x6144, 0x6817, 0x7387, 0x9686, 0x5229, 0x540F, 0x5C65 },
	/* 0x121 */ { 0x6613, 0x674E, 0x68A8, 0x6CE5, 0x7406, 0x75E2, 0x7F79, 0x88CF, 0x88E1, 0x91CC, 0x96E2, 0x533F, 0x6EBA, 0x541D, 0x71D0, 0x7498 },
	/* 0x122 */ { 0x85FA, 0x96A3, 0x9C57, 0x9E9F, 0x6797, 0x6DCB, 0x81E8, 0x7ACB, 0x7B20, 0x7C92, 0x72C0, 0x7099, 0x8B58, 0x4EC0, 0x8336, 0x523A },
	/* 0x123 */ { 0x5207, 0x5EA6, 0x62D3, 0x7CD6, 0x5B85, 0x6D1E, 0x66B4, 0x8F3B, 0x884C, 0x964D, 0x898B, 0x5ED3, 0x5140, 0x55C0, 0x0000, 0x0000 },
	/* 0x124 */ { 0x585A, 0x0000, 0x6674, 0x0000, 0x0000, 0x51DE, 0x732A, 0x76CA, 0x793C, 0x795E, 0x7965, 0x798F, 0x9756, 0x7CBE, 0x7FBD, 0x0000 },
	/* 0x125 */ { 0x8612, 0x0000, 0x8AF8, 0x0000, 0x0000, 0x9038, 0x90FD, 0x0000, 0x0000, 0x0000, 0x98EF, 0x98FC, 0x9928, 0x9DB4, 0x90DE, 0x96B7 },
	/* 0x126 */ { 0x4FAE, 0x50E7, 0x514D, 0x52C9, 0x52E4, 0x5351, 0x559D, 0x5606, 0x5668, 0x5840, 0x58A8, 0x5C64, 0x5C6E, 0x6094, 0x6168, 0x618E },
	/* 0x127 */ { 0x61F2, 0x654F, 0x65E2, 0x6691, 0x6885, 0x6D77, 0x6E1A, 0x6F22, 0x716E, 0x722B, 0x7422, 0x7891, 0x793E, 0x7949, 0x7948, 0x7950 },
	/* 0x128 */ { 0x7956, 0x795D, 0x798D, 0x798E, 0x7A40, 0x7A81, 0x7BC0, 0x7DF4, 0x7E09, 0x7E41, 0x7F72, 0x8005, 0x81ED, 0x8279, 0x8279, 0x8457 },
	/* 0x129 */ { 0x8910, 0x8996, 0x8B01, 0x8B39, 0x8CD3, 0x8D08, 0x8FB6, 0x9038, 0x96E3, 0x97FF, 0x983B, 0x6075, 0xE216, 0x8218, 0xFFFF, 0xFFFF },
	/* 0x12A */ { 0x4E26, 0x51B5, 0x5168, 0x4F80, 0x5145, 0x5180, 0x52C7, 0x52FA, 0x559D, 0x5555, 0x5599, 0x55E2, 0x585A, 0x58B3, 0x5944, 0x5954 },
	/* 0x12B */ { 0x5A62, 0x5B28, 0x5ED2, 0x5ED9, 0x5F69, 0x5FAD, 0x60D8, 0x614E, 0x6108, 0x618E, 0x6160, 0x61F2, 0x6234, 0x63C4, 0x641C, 0x6452 },
	/* 0x12C */ { 0x6556, 0x6674, 0x6717, 0x671B, 0x6756, 0x6B79, 0x6BBA, 0x6D41, 0x6EDB, 0x6ECB, 0x6F22, 0x701E, 0x716E, 0x77A7, 0x7235, 0x72AF },
	/* 0x12D */ { 0x732A, 0x7471, 0x7506, 0x753B, 0x761D, 0x761F, 0x76CA, 0x76DB, 0x76F4, 0x774A, 0x7740, 0x78CC, 0x7AB1, 0x7BC0, 0x7C7B, 0x7D5B },
	/* 0x12E */ { 0x7DF4, 0x7F3E, 0x8005, 0x8352, 0x83EF, 0x8779, 0x8941, 0x8986, 0x8996, 0x8ABF, 0x8AF8, 0x8ACB, 0x8B01, 0x8AFE, 0x8AED, 0x8B39 },
	/* 0x12F */ { 0x8B8A, 0x8D08, 0x8F38, 0x9072, 0x9199, 0x9276, 0x967C, 0x96E3, 0x9756, 0x97DB, 0x97FF, 0x980B, 0x983B, 0x9B12, 0x9F9C, 0xE217 },
	/* 0x130 */ { 0xE218, 0xE219, 0x3B9D, 0x4018, 0x4039, 0xE21A, 0xE21B, 0xE21C, 0x9F43, 0x9F8E, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x131 */ { 0xBADB, 0xBADC, 0xBADD, 0xC8DA, 0xC8DB, 0xBADE, 0xBADF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x132 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xBAE0, 0xBAE1, 0xBAE2, 0xBAE3, 0xBAE4, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xB2E5, 0xAD1A, 0xB2E6 },
	/* 0x133 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xB2E7, 0xB2E8, 0xC0DC, 0xC0DD, 0xB2E9, 0xB2EA },
	/* 0x134 */ { 0xB2EB, 0xB2EC, 0xB2ED, 0xB2EE, 0xB2EF, 0xB2F0, 0xB2F1, 0xFFFF, 0xB2F2, 0xB2F3, 0xB2F4, 0xB2F5, 0xB2F6, 0xFFFF, 0xB2F7, 0xFFFF },
	/* 0x135 */ { 0xB2F8, 0xB2F9, 0xFFFF, 0xB2FA, 0xB2FB, 0xFFFF, 0xB2FC, 0xB2FD, 0xB2FE, 0xB2FF, 0xB300, 0xB301, 0xB302, 0xB303, 0xB304, 0x0000 },
	/* 0x136 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADE6, 0xADE6 },
	/* 0x137 */ { 0x0000, 0xEA1D, 0xEA1E, 0xEA1F, 0xEA20, 0xEA21, 0xEA22, 0xEA23, 0xEA24, 0xEA25, 0xEA26, 0xEA27, 0xEA28, 0xEA29, 0xEA2A, 0xEA2B },
	/* 0x138 */ { 0xEA2C, 0xEA2D, 0xEA2E, 0xEA2F, 0xEA30, 0xEA31, 0xEA32, 0xEA33, 0xEA34, 0xEA35, 0xEA36, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x139 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0xFFFF, 0xFFFF },
	/* 0x13A */ { 0xADDC, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x13B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x13C */ { 0xEA37, 0xEA38, 0xEA39, 0xEA3A, 0xEA3B, 0xEA3C, 0xEA3D, 0xEA3E, 0xEA3F, 0xEA40, 0xEA41, 0xEA42, 0xEA43, 0xEA44, 0xEA45, 0xEA46 },
	/* 0x13D */ { 0xEA47, 0xEA48, 0xEA49, 0xEA4A, 0xEA4B, 0xEA4C, 0xEA4D, 0xEA4E, 0xEA4F, 0xEA50, 0xEA51, 0xEA52, 0xEA53, 0xEA54, 0xEA55, 0xEA56 },
	/* 0x13E */ { 0xEA57, 0xEA58, 0xEA59, 0xEA5A, 0xEA5B, 0xEA5C, 0xEA5D, 0xEA5E, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x13F */ { 0xEA5F, 0xEA60, 0xEA61, 0xEA62, 0xEA63, 0xEA64, 0xEA65, 0xEA66, 0xEA67, 0xEA68, 0xEA69, 0xEA6A, 0xEA6B, 0xEA6C, 0xEA6D, 0xEA6E },
	/* 0x140 */ { 0xEA6F, 0xEA70, 0xEA71, 0xEA72, 0xEA73, 0xEA74, 0xEA75, 0xEA76, 0xEA77, 0xEA78, 0xEA79, 0xEA7A, 0xEA7B, 0xEA7C, 0xEA7D, 0xEA7E },
	/* 0x141 */ { 0xEA7F, 0xEA80, 0xEA81, 0xEA82, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x142 */ { 0xEA83, 0xEA84, 0xEA85, 0xEA86, 0xEA87, 0xEA88, 0xEA89, 0xEA8A, 0xEA8B, 0xEA8C, 0xEA8D, 0xFFFF, 0xEA8E, 0xEA8F, 0xEA90, 0xEA91 },
	/* 0x143 */ { 0xEA92, 0xEA93, 0xEA94, 0xEA95, 0xEA96, 0xEA97, 0xEA98, 0xEA99, 0xEA9A, 0xEA9B, 0xEA9C, 0xFFFF, 0xEA9D, 0xEA9E, 0xEA9F, 0xEAA0 },
	/* 0x144 */ { 0xEAA1, 0xEAA2, 0xEAA3, 0xFFFF, 0xEAA4, 0xEAA5, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x145 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xF000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x146 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xF003, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x147 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xADDC, 0x0000, 0xADE6 },
	/* 0x148 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xADE6, 0xAD01, 0xADDC, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAD09 },
	/* 0x149 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADDC, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x14A */ { 0xEAA6, 0xEAA7, 0xEAA8, 0xEAA9, 0xEAAA, 0xEAAB, 0xEAAC, 0xEAAD, 0xEAAE, 0xEAAF, 0xEAB0, 0xEAB1, 0xEAB2, 0xEAB3, 0xEAB4, 0xEAB5 },
	/* 0x14B */ { 0xEAB6, 0xEAB7, 0xEAB8, 0xEAB9, 0xEABA, 0xEABB, 0xEABC, 0xEABD, 0xEABE, 0xEABF, 0xEAC0, 0xEAC1, 0xEAC2, 0xEAC3, 0xEAC4, 0xEAC5 },
	/* 0x14C */ { 0xEAC6, 0xEAC7, 0xEAC8, 0xEAC9, 0xEACA, 0xEACB, 0xEACC, 0xEACD, 0xEACE, 0xEACF, 0xEAD0, 0xEAD1, 0xEAD2, 0xEAD3, 0xEAD4, 0xEAD5 },
	/* 0x14D */ { 0xEAD6, 0xEAD7, 0xEAD8, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x14E */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x14F */ { 0xEAD9, 0xEADA, 0xEADB, 0xEADC, 0xEADD, 0xEADE, 0xEADF, 0xEAE0, 0xEAE1, 0xEAE2, 0xEAE3, 0xEAE4, 0xEAE5, 0xEAE6, 0xEAE7, 0xEAE8 },
	/* 0x150 */ { 0xEAE9, 0xEAEA, 0xEAEB, 0xEAEC, 0xEAED, 0xEAEE, 0xFFFF, 0xFFFF, 0xFFFF, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000 },
	/* 0x151 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xADE6, 0xADE6, 0x0000, 0xFFFF, 0xFFFF },
	/* 0x152 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0x0000, 0xADDC, 0xADDC, 0xADDC },
	/* 0x153 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0xADDC, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADE6, 0xADDC, 0xADDC, 0xADDC },
	/* 0x154 */ { 0xADDC, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x155 */ { 0x0000, 0x0000, 0xADE6, 0xADDC, 0xADE6, 0xADDC, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x156 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF },
	/* 0x157 */ { 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xAD09 },
	/* 0x158 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xF006, 0x0000, 0xF009, 0x0000, 0x0000, 0x0000 },
	/* 0x159 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xF00C, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x15A */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x15B */ { 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x15C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xF00F, 0xF012 },
	/* 0x15D */ { 0x0000, 0x0000, 0x0000, 0xAD09, 0xAD09, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x15E */ { 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x15F */ { 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x160 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x161 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD07, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x162 */ { 0x0000, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xAD07, 0xAD07, 0x0000, 0x0000, 0x0000 },
	/* 0x163 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xF015, 0xF018, 0xAD09, 0xFFFF, 0xFFFF },
	/* 0x164 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x165 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x166 */ { 0x0000, 0x0000, 0x0000, 0xF01B, 0x0000, 0xF01E, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0xFFFF, 0xFFFF, 0xF021, 0xFFFF },
	/* 0x167 */ { 0x0000, 0xF024, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x168 */ { 0x0000, 0xFFFF, 0x0000, 0xFFFF, 0xFFFF, 0xF027, 0xFFFF, 0xF02A, 0xF02D, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0xAD09, 0xAD09 },
	/* 0x169 */ { 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x16A */ { 0x0000, 0x0000, 0xAD09, 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x16B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0xADE6, 0x0000 },
	/* 0x16C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xF030, 0xF033, 0x0000, 0xF036, 0x0000 },
	/* 0x16D */ { 0x0000, 0x0000, 0xAD09, 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x16E */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xF039, 0xF03C, 0x0000, 0x0000, 0x0000, 0xAD09 },
	/* 0x16F */ { 0xAD07, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x170 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09 },
	/* 0x171 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xAD07, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x172 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x173 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xAD07, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x174 */ { 0xEAEF, 0xEAF0, 0xEAF1, 0xEAF2, 0xEAF3, 0xEAF4, 0xEAF5, 0xEAF6, 0xEAF7, 0xEAF8, 0xEAF9, 0xEAFA, 0xEAFB, 0xEAFC, 0xEAFD, 0xEAFE },
	/* 0x175 */ { 0xEAFF, 0xEB00, 0xEB01, 0xEB02, 0xEB03, 0xEB04, 0xEB05, 0xEB06, 0xEB07, 0xEB08, 0xEB09, 0xEB0A, 0xEB0B, 0xEB0C, 0xEB0D, 0xEB0E },
	/* 0x176 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0xF03F, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xAD09, 0xAD09, 0x0000 },
	/* 0x177 */ { 0x0000, 0x0000, 0x0000, 0xAD07, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x178 */ { 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x179 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x17A */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x17B */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x17C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09 },
	/* 0x17D */ { 0x0000, 0x0000, 0xAD07, 0x0000, 0xAD09, 0xAD09, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x17E */ { 0x0000, 0x0000, 0xFFFF, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x17F */ { 0x0000, 0xAD09, 0xAD09, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x180 */ { 0x0000, 0xF042, 0xF045, 0xF048, 0xF04B, 0xF04E, 0xF051, 0xF055, 0xF059, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xAD09 },
	/* 0x181 */ { 0xAD01, 0xAD01, 0xAD01, 0xAD01, 0xAD01, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x182 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x183 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xF05D, 0xF060, 0xF063, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x184 */ { 0xEB0F, 0xEB10, 0xEB11, 0xEB12, 0xEB13, 0xEB14, 0xEB15, 0xEB16, 0xEB17, 0xEB18, 0xEB19, 0xEB1A, 0xEB1B, 0xEB1C, 0xEB1D, 0xEB1E },
	/* 0x185 */ { 0xEB1F, 0xEB20, 0xEB21, 0xEB22, 0xEB23, 0xEB24, 0xEB25, 0xEB26, 0xEB27, 0xEB28, 0xEB29, 0xEB2A, 0xEB2B, 0xEB2C, 0xEB2D, 0xEB2E },
	/* 0x186 */ { 0xAD06, 0xAD06, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x187 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF, 0x0000, 0x0000, 0xAD01, 0x0000 },
	/* 0x188 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xF067, 0xF06A },
	/* 0x189 */ { 0xF06D, 0xF071, 0xF075, 0xF079, 0xF07D, 0xADD8, 0xADD8, 0xAD01, 0xAD01, 0xAD01, 0x0000, 0x0000, 0x0000, 0xADE2, 0xADD8, 0xADD8 },
	/* 0x18A */ { 0xADD8, 0xADD8, 0xADD8, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC },
	/* 0x18B */ { 0xADDC, 0xADDC, 0xADDC, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADDC, 0xADDC, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x18C */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000 },
	/* 0x18D */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xF081, 0xF084, 0xF087, 0xF08B, 0xF08F },
	/* 0x18E */ { 0xF093, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x18F */ { 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x190 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x191 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x192 */ { 0xADE6, 0xADE6, 0xFFFF, 0xADE6, 0xADE6, 0xFFFF, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x193 */ { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xADE6 },
	/* 0x194 */ { 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xFFFF, 0xFFFF },
	/* 0x195 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xFFFF },
	/* 0x196 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6 },
	/* 0x197 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE8, 0xADE8, 0xADDC, 0xADE6 },
	/* 0x198 */ { 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADDC },
	/* 0x199 */ { 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xADDC, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x19A */ { 0xEB2F, 0xEB30, 0xEB31, 0xEB32, 0xEB33, 0xEB34, 0xEB35, 0xEB36, 0xEB37, 0xEB38, 0xEB39, 0xEB3A, 0xEB3B, 0xEB3C, 0xEB3D, 0xEB3E },
	/* 0x19B */ { 0xEB3F, 0xEB40, 0xEB41, 0xEB42, 0xEB43, 0xEB44, 0xEB45, 0xEB46, 0xEB47, 0xEB48, 0xEB49, 0xEB4A, 0xEB4B, 0xEB4C, 0xEB4D, 0xEB4E },
	/* 0x19C */ { 0xEB4F, 0xEB50, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000 },
	/* 0x19D */ { 0x0000, 0x0000, 0x0000, 0x0000, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xADE6, 0xAD07, 0x0000, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF },
	/* 0x19E */ { 0x4E3D, 0x4E38, 0x4E41, 0xE351, 0x4F60, 0x4FAE, 0x4FBB, 0x5002, 0x507A, 0x5099, 0x50E7, 0x50CF, 0x349E, 0xE352, 0x514D, 0x5154 },
	/* 0x19F */ { 0x5164, 0x5177, 0xE353, 0x34B9, 0x5167, 0x518D, 0xE354, 0x5197, 0x51A4, 0x4ECC, 0x51AC, 0x51B5, 0xE355, 0x51F5, 0x5203, 0x34DF },
	/* 0x1A0 */ { 0x523B, 0x5246, 0x5272, 0x5277, 0x3515, 0x52C7, 0x52C9, 0x52E4, 0x52FA, 0x5305, 0x5306, 0x5317, 0x5349, 0x5351, 0x535A, 0x5373 },
	/* 0x1A1 */ { 0x537D, 0x537F, 0x537F, 0x537F, 0xE356, 0x7070, 0x53CA, 0x53DF, 0xE357, 0x53EB, 0x53F1, 0x5406, 0x549E, 0x5438, 0x5448, 0x5468 },
	/* 0x1A2 */ { 0x54A2, 0x54F6, 0x5510, 0x5553, 0x5563, 0x5584, 0x5584, 0x5599, 0x55AB, 0x55B3, 0x55C2, 0x5716, 0x5606, 0x5717, 0x5651, 0x5674 },
	/* 0x1A3 */ { 0x5207, 0x58EE, 0x57CE, 0x57F4, 0x580D, 0x578B, 0x5832, 0x5831, 0x58AC, 0xE358, 0x58F2, 0x58F7, 0x5906, 0x591A, 0x5922, 0x5962 },
	/* 0x1A4 */ { 0xE359, 0xE35A, 0x59EC, 0x5A1B, 0x5A27, 0x59D8, 0x5A66, 0x36EE, 0x36FC, 0x5B08, 0x5B3E, 0x5B3E, 0xE35B, 0x5BC3, 0x5BD8, 0x5BE7 },
	/* 0x1A5 */ { 0x5BF3, 0xE35C, 0x5BFF, 0x5C06, 0x5F53, 0x5C22, 0x3781, 0x5C60, 0x5C6E, 0x5CC0, 0x5C8D, 0xE35D, 0x5D43, 0xE35E, 0x5D6E, 0x5D6B },
	/* 0x1A6 */ { 0x5D7C, 0x5DE1, 0x5DE2, 0x382F, 0x5DFD, 0x5E28, 0x5E3D, 0x5E69, 0x3862, 0xE35F, 0x387C, 0x5EB0, 0x5EB3, 0x5EB6, 0x5ECA, 0xE360 },
	/* 0x1A7 */ { 0x5EFE, 0xE361, 0xE362, 0x8201, 0x5F22, 0x5F22, 0x38C7, 0xE363, 0xE364, 0x5F62, 0x5F6B, 0x38E3, 0x5F9A, 0x5FCD, 0x5FD7, 0x5FF9 },
	/* 0x1A8 */ { 0x6081, 0x393A, 0x391C, 0x6094, 0xE365, 0x60C7, 0x6148, 0x614C, 0x614E, 0x614C, 0x617A, 0x618E, 0x61B2, 0x61A4, 0x61AF, 0x61DE },
	/* 0x1A9 */ { 0x61F2, 0x61F6, 0x6210, 0x621B, 0x625D, 0x62B1, 0x62D4, 0x6350, 0xE366, 0x633D, 0x62FC, 0x6368, 0x6383, 0x63E4, 0xE367, 0x6422 },
	/* 0x1AA */ { 0x63C5, 0x63A9, 0x3A2E, 0x6469, 0x647E, 0x649D, 0x6477, 0x3A6C, 0x654F, 0x656C, 0xE368, 0x65E3, 0x66F8, 0x6649, 0x3B19, 0x6691 },
	/* 0x1AB */ { 0x3B08, 0x3AE4, 0x5192, 0x5195, 0x6700, 0x669C, 0x80AD, 0x43D9, 0x6717, 0x671B, 0x6721, 0x675E, 0x6753, 0xE369, 0x3B49, 0x67FA },
	/* 0x1AC */ { 0x6785, 0x6852, 0x6885, 0xE36A, 0x688E, 0x681F, 0x6914, 0x3B9D, 0x6942, 0x69A3, 0x69EA, 0x6AA8, 0xE36B, 0x6ADB, 0x3C18, 0x6B21 },
	/* 0x1AD */ { 0xE36C, 0x6B54, 0x3C4E, 0x6B72, 0x6B9F, 0x6BBA, 0x6BBB, 0xE36D, 0xE36E, 0xE36F, 0x6C4E, 0xE370, 0x6CBF, 0x6CCD, 0x6C67, 0x6D16 },
	/* 0x1AE */ { 0x6D3E, 0x6D77, 0x6D41, 0x6D69, 0x6D78, 0x6D85, 0xE371, 0x6D34, 0x6E2F, 0x6E6E, 0x3D33, 0x6ECB, 0x6EC7, 0xE372, 0x6DF9, 0x6F6E },
	/* 0x1AF */ { 0xE373, 0xE374, 0x6FC6, 0x7039, 0x701E, 0x701B, 0x3D96, 0x704A, 0x707D, 0x7077, 0x70AD, 0xE375, 0x7145, 0xE376, 0x719C, 0xE377 },
	/* 0x1B0 */ { 0x7228, 0x7235, 0x7250, 0xE378, 0x7280, 0x7295, 0xE379, 0xE37A, 0x737A, 0x738B, 0x3EAC, 0x73A5, 0x3EB8, 0x3EB8, 0x7447, 0x745C },
	/* 0x1B1 */ { 0x7471, 0x7485, 0x74CA, 0x3F1B, 0x7524, 0xE37B, 0x753E, 0xE37C, 0x7570, 0xE37D, 0x7610, 0xE37E, 0xE37F, 0xE380, 0x3FFC, 0x4008 },
	/* 0x1B2 */ { 0x76F4, 0xE381, 0xE382, 0xE383, 0xE384, 0x771E, 0x771F, 0x771F, 0x774A, 0x4039, 0x778B, 0x4046, 0x4096, 0xE385, 0x784E, 0x788C },
	/* 0x1B3 */ { 0x78CC, 0x40E3, 0xE386, 0x7956, 0xE387, 0xE388, 0x798F, 0x79EB, 0x412F, 0x7A40, 0x7A4A, 0x7A4F, 0xE389, 0xE38A, 0xE38B, 0x7AEE },
	/* 0x1B4 */ { 0x4202, 0xE38C, 0x7BC6, 0x7BC9, 0x4227, 0xE38D, 0x7CD2, 0x42A0, 0x7CE8, 0x7CE3, 0x7D00, 0xE38E, 0x7D63, 0x4301, 0x7DC7, 0x7E02 },
	/* 0x1B5 */ { 0x7E45, 0x4334, 0xE38F, 0xE390, 0x4359, 0xE391, 0x7F7A, 0xE392, 0x7F95, 0x7FFA, 0x8005, 0xE393, 0xE394, 0x8060, 0xE395, 0x8070 },
	/* 0x1B6 */ { 0xE396, 0x43D5, 0x80B2, 0x8103, 0x440B, 0x813E, 0x5AB5, 0xE397, 0xE398, 0xE399, 0xE39A, 0x8201, 0x8204, 0x8F9E, 0x446B, 0x8291 },
	/* 0x1B7 */ { 0x828B, 0x829D, 0x52B3, 0x82B1, 0x82B3, 0x82BD, 0x82E6, 0xE39B, 0x82E5, 0x831D, 0x8363, 0x83AD, 0x8323, 0x83BD, 0x83E7, 0x8457 },
	/* 0x1B8 */ { 0x8353, 0x83CA, 0x83CC, 0x83DC, 0xE39C, 0xE39D, 0xE39E, 0x452B, 0x84F1, 0x84F3, 0x8516, 0xE39F, 0x8564, 0xE3A0, 0x455D, 0x4561 },
	/* 0x1B9 */ { 0xE3A1, 0xE3A2, 0x456B, 0x8650, 0x865C, 0x8667, 0x8669, 0x86A9, 0x8688, 0x870E, 0x86E2, 0x8779, 0x8728, 0x876B, 0x8786, 0x45D7 },
	/* 0x1BA */ { 0x87E1, 0x8801, 0x45F9, 0x8860, 0x8863, 0xE3A3, 0x88D7, 0x88DE, 0x4635, 0x88FA, 0x34BB, 0xE3A4, 0xE3A5, 0x46BE, 0x46C7, 0x8AA0 },
	/* 0x1BB */ { 0x8AED, 0x8B8A, 0x8C55, 0xE3A6, 0x8CAB, 0x8CC1, 0x8D1B, 0x8D77, 0xE3A7, 0xE3A8, 0x8DCB, 0x8DBC, 0x8DF0, 0xE3A9, 0x8ED4, 0x8F38 },
	/* 0x1BC */ { 0xE3AA, 0xE3AB, 0x9094, 0x90F1, 0x9111, 0xE3AC, 0x911B, 0x9238, 0x92D7, 0x92D8, 0x927C, 0x93F9, 0x9415, 0xE3AD, 0x958B, 0x4995 },
	/* 0x1BD */ { 0x95B7, 0xE3AE, 0x49E6, 0x96C3, 0x5DB2, 0x9723, 0xE3AF, 0xE3B0, 0x4A6E, 0x4A76, 0x97E0, 0xE3B1, 0x4AB2, 0xE3B2, 0x980B, 0x980B },
	/* 0x1BE */ { 0x9829, 0xE3B3, 0x98E2, 0x4B33, 0x9929, 0x99A7, 0x99C2, 0x99FE, 0x4BCE, 0xE3B4, 0x9B12, 0x9C40, 0x9CFD, 0x4CCE, 0x4CED, 0x9D67 },
	/* 0x1BF */ { 0xE3B5, 0x4CF8, 0xE3B6, 0xE3B7, 0xE3B8, 0x9EBB, 0x4D56, 0x9EF9, 0x9EFE, 0x9F05, 0x9F0F, 0x9F16, 0x9F3B, 0xE3B9, 0xFFFF, 0xFFFF }
};
static uint16_t nfU16InvMasks[139] = {
	/* 0x000 */ 0x7800, 0xFFE0, 0x4000, 0xFFFC, 0x8000, 0xF800, 0x6010, 0x0006, 0x0200, 0x0030, 0x7811, 0x003F, 0xFF80, 0x4011, 0x0004, 0xFFFE,
	/* 0x010 */ 0x01FC, 0x6011, 0xFF00, 0x3813, 0x38E7, 0x3C00, 0xFF7E, 0x2000, 0x0002, 0x007F, 0x9F9F, 0xFFF1, 0x000F, 0x0011, 0x0380, 0xD004,
	/* 0x020 */ 0xFFE3, 0x0001, 0xF000, 0x0829, 0x0050, 0x0C00, 0xC200, 0xC280, 0x80C2, 0x00C2, 0x0080, 0xE000, 0xFC00, 0xFE00, 0xFFF0, 0xFFF2,
	/* 0x030 */ 0xFFC0, 0x000E, 0xC000, 0x3800, 0x1C00, 0x0020, 0x000C, 0x0040, 0xDF40, 0x7F00, 0x8080, 0x0400, 0x001F, 0x7FC0, 0x07FF, 0x8181,
	/* 0x040 */ 0xFF81, 0x0780, 0xFFF8, 0x0007, 0x0003, 0x0008, 0xF080, 0x6000, 0x0303, 0xE303, 0xC1FF, 0x1000, 0x4800, 0x0078, 0x0070, 0x1FF0,
	/* 0x050 */ 0x00F0, 0x7FF0, 0xE404, 0xF802, 0x02C0, 0x6E40, 0x07C8, 0x7000, 0x7C00, 0x0F00, 0x0110, 0x01C0, 0x00C0, 0x00F8, 0xE1FC, 0x01FF,
	/* 0x060 */ 0x03F8, 0x3FC0, 0xDFF8, 0x4280, 0x1F7E, 0xFFF9, 0x1800, 0x7FF8, 0x0D80, 0x0090, 0x0300, 0x0100, 0x0480, 0x4B80, 0x0240, 0x7FFC,
	/* 0x070 */ 0x1F00, 0x7FFF, 0x9010, 0xFFFB, 0xFFD8, 0xFF0F, 0x0180, 0x219B, 0x1400, 0x0010, 0x1840, 0x2020, 0x8400, 0x03A0, 0x3000, 0xF81F,
	/* 0x080 */ 0x9080, 0x0060, 0x0169, 0xF508, 0x157B, 0x5569, 0x0869, 0xA108, 0x0411, 0x3F80, 0xFFFD
};
static uint16_t nfU16Seq2[773][2] = {
	/* 0x000 */ {0x0041, 0x0300}, {0x0041, 0x0301}, {0x0041, 0x0302}, {0x0041, 0x0303}, {0x0041, 0x0308}, {0x0041, 0x030A},
	/* 0x006 */ {0x0043, 0x0327}, {0x0045, 0x0300}, {0x0045, 0x0301}, {0x0045, 0x0302}, {0x0045, 0x0308}, {0x0049, 0x0300},
	/* 0x00C */ {0x0049, 0x0301}, {0x0049, 0x0302}, {0x0049, 0x0308}, {0x004E, 0x0303}, {0x004F, 0x0300}, {0x004F, 0x0301},
	/* 0x012 */ {0x004F, 0x0302}, {0x004F, 0x0303}, {0x004F, 0x0308}, {0x0055, 0x0300}, {0x0055, 0x0301}, {0x0055, 0x0302},
	/* 0x018 */ {0x0055, 0x0308}, {0x0059, 0x0301}, {0x0073, 0x0073}, {0x0061, 0x0300}, {0x0061, 0x0301}, {0x0061, 0x0302},
	/* 0x01E */ {0x0061, 0x0303}, {0x0061, 0x0308}, {0x0061, 0x030A}, {0x0063, 0x0327}, {0x0065, 0x0300}, {0x0065, 0x0301},
	/* 0x024 */ {0x0065, 0x0302}, {0x0065, 0x0308}, {0x0069, 0x0300}, {0x0069, 0x0301}, {0x0069, 0x0302}, {0x0069, 0x0308},
	/* 0x02A */ {0x006E, 0x0303}, {0x006F, 0x0300}, {0x006F, 0x0301}, {0x006F, 0x0302}, {0x006F, 0x0303}, {0x006F, 0x0308},
	/* 0x030 */ {0x0075, 0x0300}, {0x0075, 0x0301}, {0x0075, 0x0302}, {0x0075, 0x0308}, {0x0079, 0x0301}, {0x0079, 0x0308},
	/* 0x036 */ {0x0041, 0x0304}, {0x0061, 0x0304}, {0x0041, 0x0306}, {0x0061, 0x0306}, {0x0041, 0x0328}, {0x0061, 0x0328},
	/* 0x03C */ {0x0043, 0x0301}, {0x0063, 0x0301}, {0x0043, 0x0302}, {0x0063, 0x0302}, {0x0043, 0x0307}, {0x0063, 0x0307},
	/* 0x042 */ {0x0043, 0x030C}, {0x0063, 0x030C}, {0x0044, 0x030C}, {0x0064, 0x030C}, {0x0045, 0x0304}, {0x0065, 0x0304},
	/* 0x048 */ {0x0045, 0x0306}, {0x0065, 0x0306}, {0x0045, 0x0307}, {0x0065, 0x0307}, {0x0045, 0x0328}, {0x0065, 0x0328},
	/* 0x04E */ {0x0045, 0x030C}, {0x0065, 0x030C}, {0x0047, 0x0302}, {0x0067, 0x0302}, {0x0047, 0x0306}, {0x0067, 0x0306},
	/* 0x054 */ {0x0047, 0x0307}, {0x0067, 0x0307}, {0x0047, 0x0327}, {0x0067, 0x0327}, {0x0048, 0x0302}, {0x0068, 0x0302},
	/* 0x05A */ {0x0049, 0x0303}, {0x0069, 0x0303}, {0x0049, 0x0304}, {0x0069, 0x0304}, {0x0049, 0x0306}, {0x0069, 0x0306},
	/* 0x060 */ {0x0049, 0x0328}, {0x0069, 0x0328}, {0x0049, 0x0307}, {0x004A, 0x0302}, {0x006A, 0x0302}, {0x004B, 0x0327},
	/* 0x066 */ {0x006B, 0x0327}, {0x004C, 0x0301}, {0x006C, 0x0301}, {0x004C, 0x0327}, {0x006C, 0x0327}, {0x004C, 0x030C},
	/* 0x06C */ {0x006C, 0x030C}, {0x004E, 0x0301}, {0x006E, 0x0301}, {0x004E, 0x0327}, {0x006E, 0x0327}, {0x004E, 0x030C},
	/* 0x072 */ {0x006E, 0x030C}, {0x02BC, 0x006E}, {0x004F, 0x0304}, {0x006F, 0x0304}, {0x004F, 0x0306}, {0x006F, 0x0306},
	/* 0x078 */ {0x004F, 0x030B}, {0x006F, 0x030B}, {0x0052, 0x0301}, {0x0072, 0x0301}, {0x0052, 0x0327}, {0x0072, 0x0327},
	/* 0x07E */ {0x0052, 0x030C}, {0x0072, 0x030C}, {0x0053, 0x0301}, {0x0073, 0x0301}, {0x0053, 0x0302}, {0x0073, 0x0302},
	/* 0x084 */ {0x0053, 0x0327}, {0x0073, 0x0327}, {0x0053, 0x030C}, {0x0073, 0x030C}, {0x0054, 0x0327}, {0x0074, 0x0327},
	/* 0x08A */ {0x0054, 0x030C}, {0x0074, 0x030C}, {0x0055, 0x0303}, {0x0075, 0x0303}, {0x0055, 0x0304}, {0x0075, 0x0304},
	/* 0x090 */ {0x0055, 0x0306}, {0x0075, 0x0306}, {0x0055, 0x030A}, {0x0075, 0x030A}, {0x0055, 0x030B}, {0x0075, 0x030B},
	/* 0x096 */ {0x0055, 0x0328}, {0x0075, 0x0328}, {0x0057, 0x0302}, {0x0077, 0x0302}, {0x0059, 0x0302}, {0x0079, 0x0302},
	/* 0x09C */ {0x0059, 0x0308}, {0x005A, 0x0301}, {0x007A, 0x0301}, {0x005A, 0x0307}, {0x007A, 0x0307}, {0x005A, 0x030C},
	/* 0x0A2 */ {0x007A, 0x030C}, {0x004F, 0x031B}, {0x006F, 0x031B}, {0x0055, 0x031B}, {0x0075, 0x031B}, {0x0041, 0x030C},
	/* 0x0A8 */ {0x0061, 0x030C}, {0x0049, 0x030C}, {0x0069, 0x030C}, {0x004F, 0x030C}, {0x006F, 0x030C}, {0x0055, 0x030C},
	/* 0x0AE */ {0x0075, 0x030C}, {0x00C6, 0x0304}, {0x00E6, 0x0304}, {0x0047, 0x030C}, {0x0067, 0x030C}, {0x004B, 0x030C},
	/* 0x0B4 */ {0x006B, 0x030C}, {0x004F, 0x0328}, {0x006F, 0x0328}, {0x01B7, 0x030C}, {0x0292, 0x030C}, {0x006A, 0x030C},
	/* 0x0BA */ {0x0047, 0x0301}, {0x0067, 0x0301}, {0x004E, 0x0300}, {0x006E, 0x0300}, {0x00C6, 0x0301}, {0x00E6, 0x0301},
	/* 0x0C0 */ {0x00D8, 0x0301}, {0x00F8, 0x0301}, {0x0041, 0x030F}, {0x0061, 0x030F}, {0x0041, 0x0311}, {0x0061, 0x0311},
	/* 0x0C6 */ {0x0045, 0x030F}, {0x0065, 0x030F}, {0x0045, 0x0311}, {0x0065, 0x0311}, {0x0049, 0x030F}, {0x0069, 0x030F},
	/* 0x0CC */ {0x0049, 0x0311}, {0x0069, 0x0311}, {0x004F, 0x030F}, {0x006F, 0x030F}, {0x004F, 0x0311}, {0x006F, 0x0311},
	/* 0x0D2 */ {0x0052, 0x030F}, {0x0072, 0x030F}, {0x0052, 0x0311}, {0x0072, 0x0311}, {0x0055, 0x030F}, {0x0075, 0x030F},
	/* 0x0D8 */ {0x0055, 0x0311}, {0x0075, 0x0311}, {0x0053, 0x0326}, {0x0073, 0x0326}, {0x0054, 0x0326}, {0x0074, 0x0326},
	/* 0x0DE */ {0x0048, 0x030C}, {0x0068, 0x030C}, {0x0041, 0x0307}, {0x0061, 0x0307}, {0x0045, 0x0327}, {0x0065, 0x0327},
	/* 0x0E4 */ {0x004F, 0x0307}, {0x006F, 0x0307}, {0x0059, 0x0304}, {0x0079, 0x0304}, {0x00A8, 0x0301}, {0x0391, 0x0301},
	/* 0x0EA */ {0x0395, 0x0301}, {0x0397, 0x0301}, {0x0399, 0x0301}, {0x039F, 0x0301}, {0x03A5, 0x0301}, {0x03A9, 0x0301},
	/* 0x0F0 */ {0x0399, 0x0308}, {0x03A5, 0x0308}, {0x03B1, 0x0301}, {0x03B5, 0x0301}, {0x03B7, 0x0301}, {0x03B9, 0x0301},
	/* 0x0F6 */ {0x03B9, 0x0308}, {0x03C5, 0x0308}, {0x03BF, 0x0301}, {0x03C5, 0x0301}, {0x03C9, 0x0301}, {0x03D2, 0x0301},
	/* 0x0FC */ {0x03D2, 0x0308}, {0x0415, 0x0300}, {0x0415, 0x0308}, {0x0413, 0x0301}, {0x0406, 0x0308}, {0x041A, 0x0301},
	/* 0x102 */ {0x0418, 0x0300}, {0x0423, 0x0306}, {0x0418, 0x0306}, {0x0438, 0x0306}, {0x0435, 0x0300}, {0x0435, 0x0308},
	/* 0x108 */ {0x0433, 0x0301}, {0x0456, 0x0308}, {0x043A, 0x0301}, {0x0438, 0x0300}, {0x0443, 0x0306}, {0x0474, 0x030F},
	/* 0x10E */ {0x0475, 0x030F}, {0x0416, 0x0306}, {0x0436, 0x0306}, {0x0410, 0x0306}, {0x0430, 0x0306}, {0x0410, 0x0308},
	/* 0x114 */ {0x0430, 0x0308}, {0x0415, 0x0306}, {0x0435, 0x0306}, {0x04D8, 0x0308}, {0x04D9, 0x0308}, {0x0416, 0x0308},
	/* 0x11A */ {0x0436, 0x0308}, {0x0417, 0x0308}, {0x0437, 0x0308}, {0x0418, 0x0304}, {0x0438, 0x0304}, {0x0418, 0x0308},
	/* 0x120 */ {0x0438, 0x0308}, {0x041E, 0x0308}, {0x043E, 0x0308}, {0x04E8, 0x0308}, {0x04E9, 0x0308}, {0x042D, 0x0308},
	/* 0x126 */ {0x044D, 0x0308}, {0x0423, 0x0304}, {0x0443, 0x0304}, {0x0423, 0x0308}, {0x0443, 0x0308}, {0x0423, 0x030B},
	/* 0x12C */ {0x0443, 0x030B}, {0x0427, 0x0308}, {0x0447, 0x0308}, {0x042B, 0x0308}, {0x044B, 0x0308}, {0x0565, 0x0582},
	/* 0x132 */ {0x0627, 0x0653}, {0x0627, 0x0654}, {0x0648, 0x0654}, {0x0627, 0x0655}, {0x064A, 0x0654}, {0x06D5, 0x0654},
	/* 0x138 */ {0x06C1, 0x0654}, {0x06D2, 0x0654}, {0x0928, 0x093C}, {0x0930, 0x093C}, {0x0933, 0x093C}, {0x0915, 0x093C},
	/* 0x13E */ {0x0916, 0x093C}, {0x0917, 0x093C}, {0x091C, 0x093C}, {0x0921, 0x093C}, {0x0922, 0x093C}, {0x092B, 0x093C},
	/* 0x144 */ {0x092F, 0x093C}, {0x09C7, 0x09BE}, {0x09C7, 0x09D7}, {0x09A1, 0x09BC}, {0x09A2, 0x09BC}, {0x09AF, 0x09BC},
	/* 0x14A */ {0x0A32, 0x0A3C}, {0x0A38, 0x0A3C}, {0x0A16, 0x0A3C}, {0x0A17, 0x0A3C}, {0x0A1C, 0x0A3C}, {0x0A2B, 0x0A3C},
	/* 0x150 */ {0x0B47, 0x0B56}, {0x0B47, 0x0B3E}, {0x0B47, 0x0B57}, {0x0B21, 0x0B3C}, {0x0B22, 0x0B3C}, {0x0B92, 0x0BD7},
	/* 0x156 */ {0x0BC6, 0x0BBE}, {0x0BC7, 0x0BBE}, {0x0BC6, 0x0BD7}, {0x0C46, 0x0C56}, {0x0CBF, 0x0CD5}, {0x0CC6, 0x0CD5},
	/* 0x15C */ {0x0CC6, 0x0CD6}, {0x0CC6, 0x0CC2}, {0x0D46, 0x0D3E}, {0x0D47, 0x0D3E}, {0x0D46, 0x0D57}, {0x0DD9, 0x0DCA},
	/* 0x162 */ {0x0DD9, 0x0DCF}, {0x0DD9, 0x0DDF}, {0x0F42, 0x0FB7}, {0x0F4C, 0x0FB7}, {0x0F51, 0x0FB7}, {0x0F56, 0x0FB7},
	/* 0x168 */ {0x0F5B, 0x0FB7}, {0x0F40, 0x0FB5}, {0x0FB2, 0x0F80}, {0x0FB3, 0x0F80}, {0x0F92, 0x0FB7}, {0x0F9C, 0x0FB7},
	/* 0x16E */ {0x0FA1, 0x0FB7}, {0x0FA6, 0x0FB7}, {0x0FAB, 0x0FB7}, {0x0F90, 0x0FB5}, {0x1025, 0x102E}, {0x1B05, 0x1B35},
	/* 0x174 */ {0x1B07, 0x1B35}, {0x1B09, 0x1B35}, {0x1B0B, 0x1B35}, {0x1B0D, 0x1B35}, {0x1B11, 0x1B35}, {0x1B3A, 0x1B35},
	/* 0x17A */ {0x1B3C, 0x1B35}, {0x1B3E, 0x1B35}, {0x1B3F, 0x1B35}, {0x1B42, 0x1B35}, {0x0041, 0x0325}, {0x0061, 0x0325},
	/* 0x180 */ {0x0042, 0x0307}, {0x0062, 0x0307}, {0x0042, 0x0323}, {0x0062, 0x0323}, {0x0042, 0x0331}, {0x0062, 0x0331},
	/* 0x186 */ {0x0044, 0x0307}, {0x0064, 0x0307}, {0x0044, 0x0323}, {0x0064, 0x0323}, {0x0044, 0x0331}, {0x0064, 0x0331},
	/* 0x18C */ {0x0044, 0x0327}, {0x0064, 0x0327}, {0x0044, 0x032D}, {0x0064, 0x032D}, {0x0045, 0x032D}, {0x0065, 0x032D},
	/* 0x192 */ {0x0045, 0x0330}, {0x0065, 0x0330}, {0x0046, 0x0307}, {0x0066, 0x0307}, {0x0047, 0x0304}, {0x0067, 0x0304},
	/* 0x198 */ {0x0048, 0x0307}, {0x0068, 0x0307}, {0x0048, 0x0323}, {0x0068, 0x0323}, {0x0048, 0x0308}, {0x0068, 0x0308},
	/* 0x19E */ {0x0048, 0x0327}, {0x0068, 0x0327}, {0x0048, 0x032E}, {0x0068, 0x032E}, {0x0049, 0x0330}, {0x0069, 0x0330},
	/* 0x1A4 */ {0x004B, 0x0301}, {0x006B, 0x0301}, {0x004B, 0x0323}, {0x006B, 0x0323}, {0x004B, 0x0331}, {0x006B, 0x0331},
	/* 0x1AA */ {0x004C, 0x0323}, {0x006C, 0x0323}, {0x004C, 0x0331}, {0x006C, 0x0331}, {0x004C, 0x032D}, {0x006C, 0x032D},
	/* 0x1B0 */ {0x004D, 0x0301}, {0x006D, 0x0301}, {0x004D, 0x0307}, {0x006D, 0x0307}, {0x004D, 0x0323}, {0x006D, 0x0323},
	/* 0x1B6 */ {0x004E, 0x0307}, {0x006E, 0x0307}, {0x004E, 0x0323}, {0x006E, 0x0323}, {0x004E, 0x0331}, {0x006E, 0x0331},
	/* 0x1BC */ {0x004E, 0x032D}, {0x006E, 0x032D}, {0x0050, 0x0301}, {0x0070, 0x0301}, {0x0050, 0x0307}, {0x0070, 0x0307},
	/* 0x1C2 */ {0x0052, 0x0307}, {0x0072, 0x0307}, {0x0052, 0x0323}, {0x0072, 0x0323}, {0x0052, 0x0331}, {0x0072, 0x0331},
	/* 0x1C8 */ {0x0053, 0x0307}, {0x0073, 0x0307}, {0x0053, 0x0323}, {0x0073, 0x0323}, {0x0054, 0x0307}, {0x0074, 0x0307},
	/* 0x1CE */ {0x0054, 0x0323}, {0x0074, 0x0323}, {0x0054, 0x0331}, {0x0074, 0x0331}, {0x0054, 0x032D}, {0x0074, 0x032D},
	/* 0x1D4 */ {0x0055, 0x0324}, {0x0075, 0x0324}, {0x0055, 0x0330}, {0x0075, 0x0330}, {0x0055, 0x032D}, {0x0075, 0x032D},
	/* 0x1DA */ {0x0056, 0x0303}, {0x0076, 0x0303}, {0x0056, 0x0323}, {0x0076, 0x0323}, {0x0057, 0x0300}, {0x0077, 0x0300},
	/* 0x1E0 */ {0x0057, 0x0301}, {0x0077, 0x0301}, {0x0057, 0x0308}, {0x0077, 0x0308}, {0x0057, 0x0307}, {0x0077, 0x0307},
	/* 0x1E6 */ {0x0057, 0x0323}, {0x0077, 0x0323}, {0x0058, 0x0307}, {0x0078, 0x0307}, {0x0058, 0x0308}, {0x0078, 0x0308},
	/* 0x1EC */ {0x0059, 0x0307}, {0x0079, 0x0307}, {0x005A, 0x0302}, {0x007A, 0x0302}, {0x005A, 0x0323}, {0x007A, 0x0323},
	/* 0x1F2 */ {0x005A, 0x0331}, {0x007A, 0x0331}, {0x0068, 0x0331}, {0x0074, 0x0308}, {0x0077, 0x030A}, {0x0079, 0x030A},
	/* 0x1F8 */ {0x0061, 0x02BE}, {0x017F, 0x0307}, {0x0073, 0x0073}, {0x0041, 0x0323}, {0x0061, 0x0323}, {0x0041, 0x0309},
	/* 0x1FE */ {0x0061, 0x0309}, {0x0045, 0x0323}, {0x0065, 0x0323}, {0x0045, 0x0309}, {0x0065, 0x0309}, {0x0045, 0x0303},
	/* 0x204 */ {0x0065, 0x0303}, {0x0049, 0x0309}, {0x0069, 0x0309}, {0x0049, 0x0323}, {0x0069, 0x0323}, {0x004F, 0x0323},
	/* 0x20A */ {0x006F, 0x0323}, {0x004F, 0x0309}, {0x006F, 0x0309}, {0x0055, 0x0323}, {0x0075, 0x0323}, {0x0055, 0x0309},
	/* 0x210 */ {0x0075, 0x0309}, {0x0059, 0x0300}, {0x0079, 0x0300}, {0x0059, 0x0323}, {0x0079, 0x0323}, {0x0059, 0x0309},
	/* 0x216 */ {0x0079, 0x0309}, {0x0059, 0x0303}, {0x0079, 0x0303}, {0x03B1, 0x0313}, {0x03B1, 0x0314}, {0x0391, 0x0313},
	/* 0x21C */ {0x0391, 0x0314}, {0x03B5, 0x0313}, {0x03B5, 0x0314}, {0x0395, 0x0313}, {0x0395, 0x0314}, {0x03B7, 0x0313},
	/* 0x222 */ {0x03B7, 0x0314}, {0x0397, 0x0313}, {0x0397, 0x0314}, {0x03B9, 0x0313}, {0x03B9, 0x0314}, {0x0399, 0x0313},
	/* 0x228 */ {0x0399, 0x0314}, {0x03BF, 0x0313}, {0x03BF, 0x0314}, {0x039F, 0x0313}, {0x039F, 0x0314}, {0x03C5, 0x0313},
	/* 0x22E */ {0x03C5, 0x0314}, {0x03A5, 0x0314}, {0x03C9, 0x0313}, {0x03C9, 0x0314}, {0x03A9, 0x0313}, {0x03A9, 0x0314},
	/* 0x234 */ {0x03B1, 0x0300}, {0x03B1, 0x0301}, {0x03B5, 0x0300}, {0x03B5, 0x0301}, {0x03B7, 0x0300}, {0x03B7, 0x0301},
	/* 0x23A */ {0x03B9, 0x0300}, {0x03B9, 0x0301}, {0x03BF, 0x0300}, {0x03BF, 0x0301}, {0x03C5, 0x0300}, {0x03C5, 0x0301},
	/* 0x240 */ {0x03C9, 0x0300}, {0x03C9, 0x0301}, {0x03B1, 0x0306}, {0x03B1, 0x0304}, {0x03B1, 0x0345}, {0x03B1, 0x0342},
	/* 0x246 */ {0x0391, 0x0306}, {0x0391, 0x0304}, {0x0391, 0x0300}, {0x0391, 0x0301}, {0x0391, 0x0345}, {0x00A8, 0x0342},
	/* 0x24C */ {0x03B7, 0x0345}, {0x03B7, 0x0342}, {0x0395, 0x0300}, {0x0395, 0x0301}, {0x0397, 0x0300}, {0x0397, 0x0301},
	/* 0x252 */ {0x0397, 0x0345}, {0x1FBF, 0x0300}, {0x1FBF, 0x0301}, {0x1FBF, 0x0342}, {0x03B9, 0x0306}, {0x03B9, 0x0304},
	/* 0x258 */ {0x03B9, 0x0342}, {0x0399, 0x0306}, {0x0399, 0x0304}, {0x0399, 0x0300}, {0x0399, 0x0301}, {0x1FFE, 0x0300},
	/* 0x25E */ {0x1FFE, 0x0301}, {0x1FFE, 0x0342}, {0x03C5, 0x0306}, {0x03C5, 0x0304}, {0x03C1, 0x0313}, {0x03C1, 0x0314},
	/* 0x264 */ {0x03C5, 0x0342}, {0x03A5, 0x0306}, {0x03A5, 0x0304}, {0x03A5, 0x0300}, {0x03A5, 0x0301}, {0x03A1, 0x0314},
	/* 0x26A */ {0x00A8, 0x0300}, {0x00A8, 0x0301}, {0x03C9, 0x0345}, {0x03C9, 0x0342}, {0x039F, 0x0300}, {0x039F, 0x0301},
	/* 0x270 */ {0x03A9, 0x0300}, {0x03A9, 0x0301}, {0x03A9, 0x0345}, {0x0041, 0x030A}, {0x2190, 0x0338}, {0x2192, 0x0338},
	/* 0x276 */ {0x2194, 0x0338}, {0x21D0, 0x0338}, {0x21D4, 0x0338}, {0x21D2, 0x0338}, {0x2203, 0x0338}, {0x2208, 0x0338},
	/* 0x27C */ {0x220B, 0x0338}, {0x2223, 0x0338}, {0x2225, 0x0338}, {0x223C, 0x0338}, {0x2243, 0x0338}, {0x2245, 0x0338},
	/* 0x282 */ {0x2248, 0x0338}, {0x003D, 0x0338}, {0x2261, 0x0338}, {0x224D, 0x0338}, {0x003C, 0x0338}, {0x003E, 0x0338},
	/* 0x288 */ {0x2264, 0x0338}, {0x2265, 0x0338}, {0x2272, 0x0338}, {0x2273, 0x0338}, {0x2276, 0x0338}, {0x2277, 0x0338},
	/* 0x28E */ {0x227A, 0x0338}, {0x227B, 0x0338}, {0x2282, 0x0338}, {0x2283, 0x0338}, {0x2286, 0x0338}, {0x2287, 0x0338},
	/* 0x294 */ {0x22A2, 0x0338}, {0x22A8, 0x0338}, {0x22A9, 0x0338}, {0x22AB, 0x0338}, {0x227C, 0x0338}, {0x227D, 0x0338},
	/* 0x29A */ {0x2291, 0x0338}, {0x2292, 0x0338}, {0x22B2, 0x0338}, {0x22B3, 0x0338}, {0x22B4, 0x0338}, {0x22B5, 0x0338},
	/* 0x2A0 */ {0x2ADD, 0x0338}, {0x304B, 0x3099}, {0x304D, 0x3099}, {0x304F, 0x3099}, {0x3051, 0x3099}, {0x3053, 0x3099},
	/* 0x2A6 */ {0x3055, 0x3099}, {0x3057, 0x3099}, {0x3059, 0x3099}, {0x305B, 0x3099}, {0x305D, 0x3099}, {0x305F, 0x3099},
	/* 0x2AC */ {0x3061, 0x3099}, {0x3064, 0x3099}, {0x3066, 0x3099}, {0x3068, 0x3099}, {0x306F, 0x3099}, {0x306F, 0x309A},
	/* 0x2B2 */ {0x3072, 0x3099}, {0x3072, 0x309A}, {0x3075, 0x3099}, {0x3075, 0x309A}, {0x3078, 0x3099}, {0x3078, 0x309A},
	/* 0x2B8 */ {0x307B, 0x3099}, {0x307B, 0x309A}, {0x3046, 0x3099}, {0x309D, 0x3099}, {0x30AB, 0x3099}, {0x30AD, 0x3099},
	/* 0x2BE */ {0x30AF, 0x3099}, {0x30B1, 0x3099}, {0x30B3, 0x3099}, {0x30B5, 0x3099}, {0x30B7, 0x3099}, {0x30B9, 0x3099},
	/* 0x2C4 */ {0x30BB, 0x3099}, {0x30BD, 0x3099}, {0x30BF, 0x3099}, {0x30C1, 0x3099}, {0x30C4, 0x3099}, {0x30C6, 0x3099},
	/* 0x2CA */ {0x30C8, 0x3099}, {0x30CF, 0x3099}, {0x30CF, 0x309A}, {0x30D2, 0x3099}, {0x30D2, 0x309A}, {0x30D5, 0x3099},
	/* 0x2D0 */ {0x30D5, 0x309A}, {0x30D8, 0x3099}, {0x30D8, 0x309A}, {0x30DB, 0x3099}, {0x30DB, 0x309A}, {0x30A6, 0x3099},
	/* 0x2D6 */ {0x30EF, 0x3099}, {0x30F0, 0x3099}, {0x30F1, 0x3099}, {0x30F2, 0x3099}, {0x30FD, 0x3099}, {0x0066, 0x0066},
	/* 0x2DC */ {0x0066, 0x0069}, {0x0066, 0x006C}, {0x0073, 0x0074}, {0x0073, 0x0074}, {0x0574, 0x0576}, {0x0574, 0x0565},
	/* 0x2E2 */ {0x0574, 0x056B}, {0x057E, 0x0576}, {0x0574, 0x056D}, {0x05D9, 0x05B4}, {0x05F2, 0x05B7}, {0x05E9, 0x05C1},
	/* 0x2E8 */ {0x05E9, 0x05C2}, {0x05D0, 0x05B7}, {0x05D0, 0x05B8}, {0x05D0, 0x05BC}, {0x05D1, 0x05BC}, {0x05D2, 0x05BC},
	/* 0x2EE */ {0x05D3, 0x05BC}, {0x05D4, 0x05BC}, {0x05D5, 0x05BC}, {0x05D6, 0x05BC}, {0x05D8, 0x05BC}, {0x05D9, 0x05BC},
	/* 0x2F4 */ {0x05DA, 0x05BC}, {0x05DB, 0x05BC}, {0x05DC, 0x05BC}, {0x05DE, 0x05BC}, {0x05E0, 0x05BC}, {0x05E1, 0x05BC},
	/* 0x2FA */ {0x05E3, 0x05BC}, {0x05E4, 0x05BC}, {0x05E6, 0x05BC}, {0x05E7, 0x05BC}, {0x05E8, 0x05BC}, {0x05E9, 0x05BC},
	/* 0x300 */ {0x05EA, 0x05BC}, {0x05D5, 0x05B9}, {0x05D1, 0x05BF}, {0x05DB, 0x05BF}, {0x05E4, 0x05BF}
};
static uint16_t nfU16Seq3[222][3] = {
	/* 0x000 */ {0x0055, 0x0308, 0x0304}, {0x0075, 0x0308, 0x0304}, {0x0055, 0x0308, 0x0301}, {0x0075, 0x0308, 0x0301},
	/* 0x004 */ {0x0055, 0x0308, 0x030C}, {0x0075, 0x0308, 0x030C}, {0x0055, 0x0308, 0x0300}, {0x0075, 0x0308, 0x0300},
	/* 0x008 */ {0x0041, 0x0308, 0x0304}, {0x0061, 0x0308, 0x0304}, {0x0041, 0x0307, 0x0304}, {0x0061, 0x0307, 0x0304},
	/* 0x00C */ {0x004F, 0x0328, 0x0304}, {0x006F, 0x0328, 0x0304}, {0x0041, 0x030A, 0x0301}, {0x0061, 0x030A, 0x0301},
	/* 0x010 */ {0x004F, 0x0308, 0x0304}, {0x006F, 0x0308, 0x0304}, {0x004F, 0x0303, 0x0304}, {0x006F, 0x0303, 0x0304},
	/* 0x014 */ {0x004F, 0x0307, 0x0304}, {0x006F, 0x0307, 0x0304}, {0x03B9, 0x0308, 0x0301}, {0x03C5, 0x0308, 0x0301},
	/* 0x018 */ {0x0CC6, 0x0CC2, 0x0CD5}, {0x0DD9, 0x0DCF, 0x0DCA}, {0x0043, 0x0327, 0x0301}, {0x0063, 0x0327, 0x0301},
	/* 0x01C */ {0x0045, 0x0304, 0x0300}, {0x0065, 0x0304, 0x0300}, {0x0045, 0x0304, 0x0301}, {0x0065, 0x0304, 0x0301},
	/* 0x020 */ {0x0045, 0x0327, 0x0306}, {0x0065, 0x0327, 0x0306}, {0x0049, 0x0308, 0x0301}, {0x0069, 0x0308, 0x0301},
	/* 0x024 */ {0x004C, 0x0323, 0x0304}, {0x006C, 0x0323, 0x0304}, {0x004F, 0x0303, 0x0301}, {0x006F, 0x0303, 0x0301},
	/* 0x028 */ {0x004F, 0x0303, 0x0308}, {0x006F, 0x0303, 0x0308}, {0x004F, 0x0304, 0x0300}, {0x006F, 0x0304, 0x0300},
	/* 0x02C */ {0x004F, 0x0304, 0x0301}, {0x006F, 0x0304, 0x0301}, {0x0052, 0x0323, 0x0304}, {0x0072, 0x0323, 0x0304},
	/* 0x030 */ {0x0053, 0x0301, 0x0307}, {0x0073, 0x0301, 0x0307}, {0x0053, 0x030C, 0x0307}, {0x0073, 0x030C, 0x0307},
	/* 0x034 */ {0x0053, 0x0323, 0x0307}, {0x0073, 0x0323, 0x0307}, {0x0055, 0x0303, 0x0301}, {0x0075, 0x0303, 0x0301},
	/* 0x038 */ {0x0055, 0x0304, 0x0308}, {0x0075, 0x0304, 0x0308}, {0x0041, 0x0302, 0x0301}, {0x0061, 0x0302, 0x0301},
	/* 0x03C */ {0x0041, 0x0302, 0x0300}, {0x0061, 0x0302, 0x0300}, {0x0041, 0x0302, 0x0309}, {0x0061, 0x0302, 0x0309},
	/* 0x040 */ {0x0041, 0x0302, 0x0303}, {0x0061, 0x0302, 0x0303}, {0x0041, 0x0323, 0x0302}, {0x0061, 0x0323, 0x0302},
	/* 0x044 */ {0x0041, 0x0306, 0x0301}, {0x0061, 0x0306, 0x0301}, {0x0041, 0x0306, 0x0300}, {0x0061, 0x0306, 0x0300},
	/* 0x048 */ {0x0041, 0x0306, 0x0309}, {0x0061, 0x0306, 0x0309}, {0x0041, 0x0306, 0x0303}, {0x0061, 0x0306, 0x0303},
	/* 0x04C */ {0x0041, 0x0323, 0x0306}, {0x0061, 0x0323, 0x0306}, {0x0045, 0x0302, 0x0301}, {0x0065, 0x0302, 0x0301},
	/* 0x050 */ {0x0045, 0x0302, 0x0300}, {0x0065, 0x0302, 0x0300}, {0x0045, 0x0302, 0x0309}, {0x0065, 0x0302, 0x0309},
	/* 0x054 */ {0x0045, 0x0302, 0x0303}, {0x0065, 0x0302, 0x0303}, {0x0045, 0x0323, 0x0302}, {0x0065, 0x0323, 0x0302},
	/* 0x058 */ {0x004F, 0x0302, 0x0301}, {0x006F, 0x0302, 0x0301}, {0x004F, 0x0302, 0x0300}, {0x006F, 0x0302, 0x0300},
	/* 0x05C */ {0x004F, 0x0302, 0x0309}, {0x006F, 0x0302, 0x0309}, {0x004F, 0x0302, 0x0303}, {0x006F, 0x0302, 0x0303},
	/* 0x060 */ {0x004F, 0x0323, 0x0302}, {0x006F, 0x0323, 0x0302}, {0x004F, 0x031B, 0x0301}, {0x006F, 0x031B, 0x0301},
	/* 0x064 */ {0x004F, 0x031B, 0x0300}, {0x006F, 0x031B, 0x0300}, {0x004F, 0x031B, 0x0309}, {0x006F, 0x031B, 0x0309},
	/* 0x068 */ {0x004F, 0x031B, 0x0303}, {0x006F, 0x031B, 0x0303}, {0x004F, 0x031B, 0x0323}, {0x006F, 0x031B, 0x0323},
	/* 0x06C */ {0x0055, 0x031B, 0x0301}, {0x0075, 0x031B, 0x0301}, {0x0055, 0x031B, 0x0300}, {0x0075, 0x031B, 0x0300},
	/* 0x070 */ {0x0055, 0x031B, 0x0309}, {0x0075, 0x031B, 0x0309}, {0x0055, 0x031B, 0x0303}, {0x0075, 0x031B, 0x0303},
	/* 0x074 */ {0x0055, 0x031B, 0x0323}, {0x0075, 0x031B, 0x0323}, {0x03B1, 0x0313, 0x0300}, {0x03B1, 0x0314, 0x0300},
	/* 0x078 */ {0x03B1, 0x0313, 0x0301}, {0x03B1, 0x0314, 0x0301}, {0x03B1, 0x0313, 0x0342}, {0x03B1, 0x0314, 0x0342},
	/* 0x07C */ {0x0391, 0x0313, 0x0300}, {0x0391, 0x0314, 0x0300}, {0x0391, 0x0313, 0x0301}, {0x0391, 0x0314, 0x0301},
	/* 0x080 */ {0x0391, 0x0313, 0x0342}, {0x0391, 0x0314, 0x0342}, {0x03B5, 0x0313, 0x0300}, {0x03B5, 0x0314, 0x0300},
	/* 0x084 */ {0x03B5, 0x0313, 0x0301}, {0x03B5, 0x0314, 0x0301}, {0x0395, 0x0313, 0x0300}, {0x0395, 0x0314, 0x0300},
	/* 0x088 */ {0x0395, 0x0313, 0x0301}, {0x0395, 0x0314, 0x0301}, {0x03B7, 0x0313, 0x0300}, {0x03B7, 0x0314, 0x0300},
	/* 0x08C */ {0x03B7, 0x0313, 0x0301}, {0x03B7, 0x0314, 0x0301}, {0x03B7, 0x0313, 0x0342}, {0x03B7, 0x0314, 0x0342},
	/* 0x090 */ {0x0397, 0x0313, 0x0300}, {0x0397, 0x0314, 0x0300}, {0x0397, 0x0313, 0x0301}, {0x0397, 0x0314, 0x0301},
	/* 0x094 */ {0x0397, 0x0313, 0x0342}, {0x0397, 0x0314, 0x0342}, {0x03B9, 0x0313, 0x0300}, {0x03B9, 0x0314, 0x0300},
	/* 0x098 */ {0x03B9, 0x0313, 0x0301}, {0x03B9, 0x0314, 0x0301}, {0x03B9, 0x0313, 0x0342}, {0x03B9, 0x0314, 0x0342},
	/* 0x09C */ {0x0399, 0x0313, 0x0300}, {0x0399, 0x0314, 0x0300}, {0x0399, 0x0313, 0x0301}, {0x0399, 0x0314, 0x0301},
	/* 0x0A0 */ {0x0399, 0x0313, 0x0342}, {0x0399, 0x0314, 0x0342}, {0x03BF, 0x0313, 0x0300}, {0x03BF, 0x0314, 0x0300},
	/* 0x0A4 */ {0x03BF, 0x0313, 0x0301}, {0x03BF, 0x0314, 0x0301}, {0x039F, 0x0313, 0x0300}, {0x039F, 0x0314, 0x0300},
	/* 0x0A8 */ {0x039F, 0x0313, 0x0301}, {0x039F, 0x0314, 0x0301}, {0x03C5, 0x0313, 0x0300}, {0x03C5, 0x0314, 0x0300},
	/* 0x0AC */ {0x03C5, 0x0313, 0x0301}, {0x03C5, 0x0314, 0x0301}, {0x03C5, 0x0313, 0x0342}, {0x03C5, 0x0314, 0x0342},
	/* 0x0B0 */ {0x03A5, 0x0314, 0x0300}, {0x03A5, 0x0314, 0x0301}, {0x03A5, 0x0314, 0x0342}, {0x03C9, 0x0313, 0x0300},
	/* 0x0B4 */ {0x03C9, 0x0314, 0x0300}, {0x03C9, 0x0313, 0x0301}, {0x03C9, 0x0314, 0x0301}, {0x03C9, 0x0313, 0x0342},
	/* 0x0B8 */ {0x03C9, 0x0314, 0x0342}, {0x03A9, 0x0313, 0x0300}, {0x03A9, 0x0314, 0x0300}, {0x03A9, 0x0313, 0x0301},
	/* 0x0BC */ {0x03A9, 0x0314, 0x0301}, {0x03A9, 0x0313, 0x0342}, {0x03A9, 0x0314, 0x0342}, {0x03B1, 0x0313, 0x0345},
	/* 0x0C0 */ {0x03B1, 0x0314, 0x0345}, {0x0391, 0x0313, 0x0345}, {0x0391, 0x0314, 0x0345}, {0x03B7, 0x0313, 0x0345},
	/* 0x0C4 */ {0x03B7, 0x0314, 0x0345}, {0x0397, 0x0313, 0x0345}, {0x0397, 0x0314, 0x0345}, {0x03C9, 0x0313, 0x0345},
	/* 0x0C8 */ {0x03C9, 0x0314, 0x0345}, {0x03A9, 0x0313, 0x0345}, {0x03A9, 0x0314, 0x0345}, {0x03B1, 0x0300, 0x0345},
	/* 0x0CC */ {0x03B1, 0x0301, 0x0345}, {0x03B1, 0x0342, 0x0345}, {0x03B7, 0x0300, 0x0345}, {0x03B7, 0x0301, 0x0345},
	/* 0x0D0 */ {0x03B7, 0x0342, 0x0345}, {0x03B9, 0x0308, 0x0300}, {0x03B9, 0x0308, 0x0301}, {0x03B9, 0x0308, 0x0342},
	/* 0x0D4 */ {0x03C5, 0x0308, 0x0300}, {0x03C5, 0x0308, 0x0301}, {0x03C5, 0x0308, 0x0342}, {0x03C9, 0x0300, 0x0345},
	/* 0x0D8 */ {0x03C9, 0x0301, 0x0345}, {0x03C9, 0x0342, 0x0345}, {0x0066, 0x0066, 0x0069}, {0x0066, 0x0066, 0x006C},
	/* 0x0DC */ {0x05E9, 0x05BC, 0x05C1}, {0x05E9, 0x05BC, 0x05C2}
};
static uint16_t nfU16SeqMisc[198] = {
	/* 0x000 */ 0x0E61, 0x0300, 0x0E61, 0x0301, 0x0E61, 0x0313, 0x0E62, 0x0308, 0x0301, 0x0812, 0x0F71, 0x0F72, 0x0812, 0x0F71, 0x0F74, 0x0812,
	/* 0x010 */ 0x0F71, 0x0F80, 0x0004, 0x03B1, 0x0313, 0x0300, 0x0345, 0x0004, 0x03B1, 0x0314, 0x0300, 0x0345, 0x0004, 0x03B1, 0x0313, 0x0301,
	/* 0x020 */ 0x0345, 0x0004, 0x03B1, 0x0314, 0x0301, 0x0345, 0x0004, 0x03B1, 0x0313, 0x0342, 0x0345, 0x0004, 0x03B1, 0x0314, 0x0342, 0x0345,
	/* 0x030 */ 0x0004, 0x0391, 0x0313, 0x0300, 0x0345, 0x0004, 0x0391, 0x0314, 0x0300, 0x0345, 0x0004, 0x0391, 0x0313, 0x0301, 0x0345, 0x0004,
	/* 0x040 */ 0x0391, 0x0314, 0x0301, 0x0345, 0x0004, 0x0391, 0x0313, 0x0342, 0x0345, 0x0004, 0x0391, 0x0314, 0x0342, 0x0345, 0x0004, 0x03B7,
	/* 0x050 */ 0x0313, 0x0300, 0x0345, 0x0004, 0x03B7, 0x0314, 0x0300, 0x0345, 0x0004, 0x03B7, 0x0313, 0x0301, 0x0345, 0x0004, 0x03B7, 0x0314,
	/* 0x060 */ 0x0301, 0x0345, 0x0004, 0x03B7, 0x0313, 0x0342, 0x0345, 0x0004, 0x03B7, 0x0314, 0x0342, 0x0345, 0x0004, 0x0397, 0x0313, 0x0300,
	/* 0x070 */ 0x0345, 0x0004, 0x0397, 0x0314, 0x0300, 0x0345, 0x0004, 0x0397, 0x0313, 0x0301, 0x0345, 0x0004, 0x0397, 0x0314, 0x0301, 0x0345,
	/* 0x080 */ 0x0004, 0x0397, 0x0313, 0x0342, 0x0345, 0x0004, 0x0397, 0x0314, 0x0342, 0x0345, 0x0004, 0x03C9, 0x0313, 0x0300, 0x0345, 0x0004,
	/* 0x090 */ 0x03C9, 0x0314, 0x0300, 0x0345, 0x0004, 0x03C9, 0x0313, 0x0301, 0x0345, 0x0004, 0x03C9, 0x0314, 0x0301, 0x0345, 0x0004, 0x03C9,
	/* 0x0A0 */ 0x0313, 0x0342, 0x0345, 0x0004, 0x03C9, 0x0314, 0x0342, 0x0345, 0x0004, 0x03A9, 0x0313, 0x0300, 0x0345, 0x0004, 0x03A9, 0x0314,
	/* 0x0B0 */ 0x0300, 0x0345, 0x0004, 0x03A9, 0x0313, 0x0301, 0x0345, 0x0004, 0x03A9, 0x0314, 0x0301, 0x0345, 0x0004, 0x03A9, 0x0313, 0x0342,
	/* 0x0C0 */ 0x0345, 0x0004, 0x03A9, 0x0314, 0x0342, 0x0345
};
static int32_t nfU32Char[954] = {
	/* 0x000 */ 0x00501, 0x00503, 0x00505, 0x00507, 0x00509, 0x0050B, 0x0050D, 0x0050F,
	/* 0x008 */ 0x00511, 0x00513, 0x00515, 0x00517, 0x00519, 0x0051B, 0x0051D, 0x0051F,
	/* 0x010 */ 0x00521, 0x00523, 0x00525, 0x00527, 0x00529, 0x0052B, 0x0052D, 0x0052F,
	/* 0x018 */ 0x00561, 0x00562, 0x00563, 0x00564, 0x00565, 0x00566, 0x00567, 0x00568,
	/* 0x020 */ 0x00569, 0x0056A, 0x0056B, 0x0056C, 0x0056D, 0x0056E, 0x0056F, 0x00570,
	/* 0x028 */ 0x00571, 0x00572, 0x00573, 0x00574, 0x00575, 0x00576, 0x00577, 0x00578,
	/* 0x030 */ 0x00579, 0x0057A, 0x0057B, 0x0057C, 0x0057D, 0x0057E, 0x0057F, 0x00580,
	/* 0x038 */ 0x00581, 0x00582, 0x00583, 0x00584, 0x00585, 0x00586, 0x02D00, 0x02D01,
	/* 0x040 */ 0x02D02, 0x02D03, 0x02D04, 0x02D05, 0x02D06, 0x02D07, 0x02D08, 0x02D09,
	/* 0x048 */ 0x02D0A, 0x02D0B, 0x02D0C, 0x02D0D, 0x02D0E, 0x02D0F, 0x02D10, 0x02D11,
	/* 0x050 */ 0x02D12, 0x02D13, 0x02D14, 0x02D15, 0x02D16, 0x02D17, 0x02D18, 0x02D19,
	/* 0x058 */ 0x02D1A, 0x02D1B, 0x02D1C, 0x02D1D, 0x02D1E, 0x02D1F, 0x02D20, 0x02D21,
	/* 0x060 */ 0x02D22, 0x02D23, 0x02D24, 0x02D25, 0x02D27, 0x02D2D, 0x013F0, 0x013F1,
	/* 0x068 */ 0x013F2, 0x013F3, 0x013F4, 0x013F5, 0x00432, 0x00434, 0x0043E, 0x00441,
	/* 0x070 */ 0x00442, 0x00442, 0x0044A, 0x00463, 0x0A64B, 0x01C8A, 0x010D0, 0x010D1,
	/* 0x078 */ 0x010D2, 0x010D3, 0x010D4, 0x010D5, 0x010D6, 0x010D7, 0x010D8, 0x010D9,
	/* 0x080 */ 0x010DA, 0x010DB, 0x010DC, 0x010DD, 0x010DE, 0x010DF, 0x010E0, 0x010E1,
	/* 0x088 */ 0x010E2, 0x010E3, 0x010E4, 0x010E5, 0x010E6, 0x010E7, 0x010E8, 0x010E9,
	/* 0x090 */ 0x010EA, 0x010EB, 0x010EC, 0x010ED, 0x010EE, 0x010EF, 0x010F0, 0x010F1,
	/* 0x098 */ 0x010F2, 0x010F3, 0x010F4, 0x010F5, 0x010F6, 0x010F7, 0x010F8, 0x010F9,
	/* 0x0A0 */ 0x010FA, 0x010FD, 0x010FE, 0x010FF, 0x01EFB, 0x01EFD, 0x01EFF, 0x0214E,
	/* 0x0A8 */ 0x02170, 0x02171, 0x02172, 0x02173, 0x02174, 0x02175, 0x02176, 0x02177,
	/* 0x0B0 */ 0x02178, 0x02179, 0x0217A, 0x0217B, 0x0217C, 0x0217D, 0x0217E, 0x0217F,
	/* 0x0B8 */ 0x02184, 0x024D0, 0x024D1, 0x024D2, 0x024D3, 0x024D4, 0x024D5, 0x024D6,
	/* 0x0C0 */ 0x024D7, 0x024D8, 0x024D9, 0x024DA, 0x024DB, 0x024DC, 0x024DD, 0x024DE,
	/* 0x0C8 */ 0x024DF, 0x024E0, 0x024E1, 0x024E2, 0x024E3, 0x024E4, 0x024E5, 0x024E6,
	/* 0x0D0 */ 0x024E7, 0x024E8, 0x024E9, 0x02C30, 0x02C31, 0x02C32, 0x02C33, 0x02C34,
	/* 0x0D8 */ 0x02C35, 0x02C36, 0x02C37, 0x02C38, 0x02C39, 0x02C3A, 0x02C3B, 0x02C3C,
	/* 0x0E0 */ 0x02C3D, 0x02C3E, 0x02C3F, 0x02C40, 0x02C41, 0x02C42, 0x02C43, 0x02C44,
	/* 0x0E8 */ 0x02C45, 0x02C46, 0x02C47, 0x02C48, 0x02C49, 0x02C4A, 0x02C4B, 0x02C4C,
	/* 0x0F0 */ 0x02C4D, 0x02C4E, 0x02C4F, 0x02C50, 0x02C51, 0x02C52, 0x02C53, 0x02C54,
	/* 0x0F8 */ 0x02C55, 0x02C56, 0x02C57, 0x02C58, 0x02C59, 0x02C5A, 0x02C5B, 0x02C5C,
	/* 0x100 */ 0x02C5D, 0x02C5E, 0x02C5F, 0x02C61, 0x0026B, 0x01D7D, 0x0027D, 0x02C68,
	/* 0x108 */ 0x02C6A, 0x02C6C, 0x00251, 0x00271, 0x00250, 0x00252, 0x02C73, 0x02C76,
	/* 0x110 */ 0x0023F, 0x00240, 0x02C81, 0x02C83, 0x02C85, 0x02C87, 0x02C89, 0x02C8B,
	/* 0x118 */ 0x02C8D, 0x02C8F, 0x02C91, 0x02C93, 0x02C95, 0x02C97, 0x02C99, 0x02C9B,
	/* 0x120 */ 0x02C9D, 0x02C9F, 0x02CA1, 0x02CA3, 0x02CA5, 0x02CA7, 0x02CA9, 0x02CAB,
	/* 0x128 */ 0x02CAD, 0x02CAF, 0x02CB1, 0x02CB3, 0x02CB5, 0x02CB7, 0x02CB9, 0x02CBB,
	/* 0x130 */ 0x02CBD, 0x02CBF, 0x02CC1, 0x02CC3, 0x02CC5, 0x02CC7, 0x02CC9, 0x02CCB,
	/* 0x138 */ 0x02CCD, 0x02CCF, 0x02CD1, 0x02CD3, 0x02CD5, 0x02CD7, 0x02CD9, 0x02CDB,
	/* 0x140 */ 0x02CDD, 0x02CDF, 0x02CE1, 0x02CE3, 0x02CEC, 0x02CEE, 0x02CF3, 0x0A641,
	/* 0x148 */ 0x0A643, 0x0A645, 0x0A647, 0x0A649, 0x0A64B, 0x0A64D, 0x0A64F, 0x0A651,
	/* 0x150 */ 0x0A653, 0x0A655, 0x0A657, 0x0A659, 0x0A65B, 0x0A65D, 0x0A65F, 0x0A661,
	/* 0x158 */ 0x0A663, 0x0A665, 0x0A667, 0x0A669, 0x0A66B, 0x0A66D, 0x0A681, 0x0A683,
	/* 0x160 */ 0x0A685, 0x0A687, 0x0A689, 0x0A68B, 0x0A68D, 0x0A68F, 0x0A691, 0x0A693,
	/* 0x168 */ 0x0A695, 0x0A697, 0x0A699, 0x0A69B, 0x0A723, 0x0A725, 0x0A727, 0x0A729,
	/* 0x170 */ 0x0A72B, 0x0A72D, 0x0A72F, 0x0A733, 0x0A735, 0x0A737, 0x0A739, 0x0A73B,
	/* 0x178 */ 0x0A73D, 0x0A73F, 0x0A741, 0x0A743, 0x0A745, 0x0A747, 0x0A749, 0x0A74B,
	/* 0x180 */ 0x0A74D, 0x0A74F, 0x0A751, 0x0A753, 0x0A755, 0x0A757, 0x0A759, 0x0A75B,
	/* 0x188 */ 0x0A75D, 0x0A75F, 0x0A761, 0x0A763, 0x0A765, 0x0A767, 0x0A769, 0x0A76B,
	/* 0x190 */ 0x0A76D, 0x0A76F, 0x0A77A, 0x0A77C, 0x01D79, 0x0A77F, 0x0A781, 0x0A783,
	/* 0x198 */ 0x0A785, 0x0A787, 0x0A78C, 0x00265, 0x0A791, 0x0A793, 0x0A797, 0x0A799,
	/* 0x1A0 */ 0x0A79B, 0x0A79D, 0x0A79F, 0x0A7A1, 0x0A7A3, 0x0A7A5, 0x0A7A7, 0x0A7A9,
	/* 0x1A8 */ 0x00266, 0x0025C, 0x00261, 0x0026C, 0x0026A, 0x0029E, 0x00287, 0x0029D,
	/* 0x1B0 */ 0x0AB53, 0x0A7B5, 0x0A7B7, 0x0A7B9, 0x0A7BB, 0x0A7BD, 0x0A7BF, 0x0A7C1,
	/* 0x1B8 */ 0x0A7C3, 0x0A794, 0x00282, 0x01D8E, 0x0A7C8, 0x0A7CA, 0x00264, 0x0A7CD,
	/* 0x1C0 */ 0x0A7D1, 0x0A7D7, 0x0A7D9, 0x0A7DB, 0x0019B, 0x0A7F6, 0x013A0, 0x013A1,
	/* 0x1C8 */ 0x013A2, 0x013A3, 0x013A4, 0x013A5, 0x013A6, 0x013A7, 0x013A8, 0x013A9,
	/* 0x1D0 */ 0x013AA, 0x013AB, 0x013AC, 0x013AD, 0x013AE, 0x013AF, 0x013B0, 0x013B1,
	/* 0x1D8 */ 0x013B2, 0x013B3, 0x013B4, 0x013B5, 0x013B6, 0x013B7, 0x013B8, 0x013B9,
	/* 0x1E0 */ 0x013BA, 0x013BB, 0x013BC, 0x013BD, 0x013BE, 0x013BF, 0x013C0, 0x013C1,
	/* 0x1E8 */ 0x013C2, 0x013C3, 0x013C4, 0x013C5, 0x013C6, 0x013C7, 0x013C8, 0x013C9,
	/* 0x1F0 */ 0x013CA, 0x013CB, 0x013CC, 0x013CD, 0x013CE, 0x013CF, 0x013D0, 0x013D1,
	/* 0x1F8 */ 0x013D2, 0x013D3, 0x013D4, 0x013D5, 0x013D6, 0x013D7, 0x013D8, 0x013D9,
	/* 0x200 */ 0x013DA, 0x013DB, 0x013DC, 0x013DD, 0x013DE, 0x013DF, 0x013E0, 0x013E1,
	/* 0x208 */ 0x013E2, 0x013E3, 0x013E4, 0x013E5, 0x013E6, 0x013E7, 0x013E8, 0x013E9,
	/* 0x210 */ 0x013EA, 0x013EB, 0x013EC, 0x013ED, 0x013EE, 0x013EF, 0x242EE, 0x2284A,
	/* 0x218 */ 0x22844, 0x233D5, 0x25249, 0x25CD0, 0x27ED3, 0x0FF41, 0x0FF42, 0x0FF43,
	/* 0x220 */ 0x0FF44, 0x0FF45, 0x0FF46, 0x0FF47, 0x0FF48, 0x0FF49, 0x0FF4A, 0x0FF4B,
	/* 0x228 */ 0x0FF4C, 0x0FF4D, 0x0FF4E, 0x0FF4F, 0x0FF50, 0x0FF51, 0x0FF52, 0x0FF53,
	/* 0x230 */ 0x0FF54, 0x0FF55, 0x0FF56, 0x0FF57, 0x0FF58, 0x0FF59, 0x0FF5A, 0x10428,
	/* 0x238 */ 0x10429, 0x1042A, 0x1042B, 0x1042C, 0x1042D, 0x1042E, 0x1042F, 0x10430,
	/* 0x240 */ 0x10431, 0x10432, 0x10433, 0x10434, 0x10435, 0x10436, 0x10437, 0x10438,
	/* 0x248 */ 0x10439, 0x1043A, 0x1043B, 0x1043C, 0x1043D, 0x1043E, 0x1043F, 0x10440,
	/* 0x250 */ 0x10441, 0x10442, 0x10443, 0x10444, 0x10445, 0x10446, 0x10447, 0x10448,
	/* 0x258 */ 0x10449, 0x1044A, 0x1044B, 0x1044C, 0x1044D, 0x1044E, 0x1044F, 0x104D8,
	/* 0x260 */ 0x104D9, 0x104DA, 0x104DB, 0x104DC, 0x104DD, 0x104DE, 0x104DF, 0x104E0,
	/* 0x268 */ 0x104E1, 0x104E2, 0x104E3, 0x104E4, 0x104E5, 0x104E6, 0x104E7, 0x104E8,
	/* 0x270 */ 0x104E9, 0x104EA, 0x104EB, 0x104EC, 0x104ED, 0x104EE, 0x104EF, 0x104F0,
	/* 0x278 */ 0x104F1, 0x104F2, 0x104F3, 0x104F4, 0x104F5, 0x104F6, 0x104F7, 0x104F8,
	/* 0x280 */ 0x104F9, 0x104FA, 0x104FB, 0x10597, 0x10598, 0x10599, 0x1059A, 0x1059B,
	/* 0x288 */ 0x1059C, 0x1059D, 0x1059E, 0x1059F, 0x105A0, 0x105A1, 0x105A3, 0x105A4,
	/* 0x290 */ 0x105A5, 0x105A6, 0x105A7, 0x105A8, 0x105A9, 0x105AA, 0x105AB, 0x105AC,
	/* 0x298 */ 0x105AD, 0x105AE, 0x105AF, 0x105B0, 0x105B1, 0x105B3, 0x105B4, 0x105B5,
	/* 0x2A0 */ 0x105B6, 0x105B7, 0x105B8, 0x105B9, 0x105BB, 0x105BC, 0x10CC0, 0x10CC1,
	/* 0x2A8 */ 0x10CC2, 0x10CC3, 0x10CC4, 0x10CC5, 0x10CC6, 0x10CC7, 0x10CC8, 0x10CC9,
	/* 0x2B0 */ 0x10CCA, 0x10CCB, 0x10CCC, 0x10CCD, 0x10CCE, 0x10CCF, 0x10CD0, 0x10CD1,
	/* 0x2B8 */ 0x10CD2, 0x10CD3, 0x10CD4, 0x10CD5, 0x10CD6, 0x10CD7, 0x10CD8, 0x10CD9,
	/* 0x2C0 */ 0x10CDA, 0x10CDB, 0x10CDC, 0x10CDD, 0x10CDE, 0x10CDF, 0x10CE0, 0x10CE1,
	/* 0x2C8 */ 0x10CE2, 0x10CE3, 0x10CE4, 0x10CE5, 0x10CE6, 0x10CE7, 0x10CE8, 0x10CE9,
	/* 0x2D0 */ 0x10CEA, 0x10CEB, 0x10CEC, 0x10CED, 0x10CEE, 0x10CEF, 0x10CF0, 0x10CF1,
	/* 0x2D8 */ 0x10CF2, 0x10D70, 0x10D71, 0x10D72, 0x10D73, 0x10D74, 0x10D75, 0x10D76,
	/* 0x2E0 */ 0x10D77, 0x10D78, 0x10D79, 0x10D7A, 0x10D7B, 0x10D7C, 0x10D7D, 0x10D7E,
	/* 0x2E8 */ 0x10D7F, 0x10D80, 0x10D81, 0x10D82, 0x10D83, 0x10D84, 0x10D85, 0x118C0,
	/* 0x2F0 */ 0x118C1, 0x118C2, 0x118C3, 0x118C4, 0x118C5, 0x118C6, 0x118C7, 0x118C8,
	/* 0x2F8 */ 0x118C9, 0x118CA, 0x118CB, 0x118CC, 0x118CD, 0x118CE, 0x118CF, 0x118D0,
	/* 0x300 */ 0x118D1, 0x118D2, 0x118D3, 0x118D4, 0x118D5, 0x118D6, 0x118D7, 0x118D8,
	/* 0x308 */ 0x118D9, 0x118DA, 0x118DB, 0x118DC, 0x118DD, 0x118DE, 0x118DF, 0x16E60,
	/* 0x310 */ 0x16E61, 0x16E62, 0x16E63, 0x16E64, 0x16E65, 0x16E66, 0x16E67, 0x16E68,
	/* 0x318 */ 0x16E69, 0x16E6A, 0x16E6B, 0x16E6C, 0x16E6D, 0x16E6E, 0x16E6F, 0x16E70,
	/* 0x320 */ 0x16E71, 0x16E72, 0x16E73, 0x16E74, 0x16E75, 0x16E76, 0x16E77, 0x16E78,
	/* 0x328 */ 0x16E79, 0x16E7A, 0x16E7B, 0x16E7C, 0x16E7D, 0x16E7E, 0x16E7F, 0x1E922,
	/* 0x330 */ 0x1E923, 0x1E924, 0x1E925, 0x1E926, 0x1E927, 0x1E928, 0x1E929, 0x1E92A,
	/* 0x338 */ 0x1E92B, 0x1E92C, 0x1E92D, 0x1E92E, 0x1E92F, 0x1E930, 0x1E931, 0x1E932,
	/* 0x340 */ 0x1E933, 0x1E934, 0x1E935, 0x1E936, 0x1E937, 0x1E938, 0x1E939, 0x1E93A,
	/* 0x348 */ 0x1E93B, 0x1E93C, 0x1E93D, 0x1E93E, 0x1E93F, 0x1E940, 0x1E941, 0x1E942,
	/* 0x350 */ 0x1E943, 0x20122, 0x2063A, 0x2051C, 0x2054B, 0x291DF, 0x20A2C, 0x20B63,
	/* 0x358 */ 0x214E4, 0x216A8, 0x216EA, 0x219C8, 0x21B18, 0x21DE4, 0x21DE6, 0x22183,
	/* 0x360 */ 0x2A392, 0x22331, 0x22331, 0x232B8, 0x261DA, 0x226D4, 0x22B0C, 0x22BF1,
	/* 0x368 */ 0x2300A, 0x233C3, 0x2346D, 0x236A3, 0x238A7, 0x23A8D, 0x21D0B, 0x23AFA,
	/* 0x370 */ 0x23CBC, 0x23D1E, 0x23ED1, 0x23F5E, 0x23F8E, 0x20525, 0x24263, 0x243AB,
	/* 0x378 */ 0x24608, 0x24735, 0x24814, 0x24C36, 0x24C92, 0x2219F, 0x24FA1, 0x24FB8,
	/* 0x380 */ 0x25044, 0x250F3, 0x250F2, 0x25119, 0x25133, 0x2541D, 0x25626, 0x2569A,
	/* 0x388 */ 0x256C5, 0x2597C, 0x25AA7, 0x25AA7, 0x25BAB, 0x25C80, 0x25F86, 0x26228,
	/* 0x390 */ 0x26247, 0x262D9, 0x2633E, 0x264DA, 0x26523, 0x265A8, 0x2335F, 0x267A7,
	/* 0x398 */ 0x267B5, 0x23393, 0x2339C, 0x26B3C, 0x26C36, 0x26D6B, 0x26CD5, 0x273CA,
	/* 0x3A0 */ 0x26F2C, 0x26FB1, 0x270D2, 0x27667, 0x278AE, 0x27966, 0x27CA8, 0x27F2F,
	/* 0x3A8 */ 0x20804, 0x208DE, 0x285D2, 0x285ED, 0x2872E, 0x28BFA, 0x28D77, 0x29145,
	/* 0x3B0 */ 0x2921A, 0x2940A, 0x29496, 0x295B6, 0x29B30, 0x2A0CE, 0x2A105, 0x2A20E,
	/* 0x3B8 */ 0x2A291, 0x2A600
};
static int32_t nfU32SeqMisc[151] = {
	/* 0x000 */ 0x00002, 0x105D2, 0x00307, 0x00002, 0x105DA, 0x00307, 0x00002, 0x11099,
	/* 0x008 */ 0x110BA, 0x00002, 0x1109B, 0x110BA, 0x00002, 0x110A5, 0x110BA, 0x00002,
	/* 0x010 */ 0x11131, 0x11127, 0x00002, 0x11132, 0x11127, 0x00002, 0x11347, 0x1133E,
	/* 0x018 */ 0x00002, 0x11347, 0x11357, 0x00002, 0x11382, 0x113C9, 0x00002, 0x11384,
	/* 0x020 */ 0x113BB, 0x00002, 0x1138B, 0x113C2, 0x00002, 0x11390, 0x113C9, 0x00002,
	/* 0x028 */ 0x113C2, 0x113C2, 0x00002, 0x113C2, 0x113B8, 0x00002, 0x113C2, 0x113C9,
	/* 0x030 */ 0x00002, 0x114B9, 0x114BA, 0x00002, 0x114B9, 0x114B0, 0x00002, 0x114B9,
	/* 0x038 */ 0x114BD, 0x00002, 0x115B8, 0x115AF, 0x00002, 0x115B9, 0x115AF, 0x00002,
	/* 0x040 */ 0x11935, 0x11930, 0x00002, 0x1611E, 0x1611E, 0x00002, 0x1611E, 0x16129,
	/* 0x048 */ 0x00002, 0x1611E, 0x1611F, 0x00002, 0x16129, 0x1611F, 0x00002, 0x1611E,
	/* 0x050 */ 0x16120, 0x00003, 0x1611E, 0x1611E, 0x1611F, 0x00003, 0x1611E, 0x16129,
	/* 0x058 */ 0x1611F, 0x00003, 0x1611E, 0x1611E, 0x16120, 0x00002, 0x16D67, 0x16D67,
	/* 0x060 */ 0x00002, 0x16D63, 0x16D67, 0x00003, 0x16D63, 0x16D67, 0x16D67, 0x00002,
	/* 0x068 */ 0x1D157, 0x1D165, 0x00002, 0x1D158, 0x1D165, 0x00003, 0x1D158, 0x1D165,
	/* 0x070 */ 0x1D16E, 0x00003, 0x1D158, 0x1D165, 0x1D16F, 0x00003, 0x1D158, 0x1D165,
	/* 0x078 */ 0x1D170, 0x00003, 0x1D158, 0x1D165, 0x1D171, 0x00003, 0x1D158, 0x1D165,
	/* 0x080 */ 0x1D172, 0x00002, 0x1D1B9, 0x1D165, 0x00002, 0x1D1BA, 0x1D165, 0x00003,
	/* 0x088 */ 0x1D1B9, 0x1D165, 0x1D16E, 0x00003, 0x1D1BA, 0x1D165, 0x1D16E, 0x00003,
	/* 0x090 */ 0x1D1B9, 0x1D165, 0x1D16F, 0x00003, 0x1D1BA, 0x1D165, 0x1D16F
};
static uint16_t nfBasicCF[1280] = {
	/* 0x000 */ 0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008, 0x0009, 0x000A, 0x000B, 0x000C, 0x000D, 0x000E, 0x000F,
	/* 0x010 */ 0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017, 0x0018, 0x0019, 0x001A, 0x001B, 0x001C, 0x001D, 0x001E, 0x001F,
	/* 0x020 */ 0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002A, 0x002B, 0x002C, 0x002D, 0x002E, 0x002F,
	/* 0x030 */ 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x003A, 0x003B, 0x003C, 0x003D, 0x003E, 0x003F,
	/* 0x040 */ 0x0040, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	/* 0x050 */ 0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007A, 0x005B, 0x005C, 0x005D, 0x005E, 0x005F,
	/* 0x060 */ 0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x006A, 0x006B, 0x006C, 0x006D, 0x006E, 0x006F,
	/* 0x070 */ 0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007A, 0x007B, 0x007C, 0x007D, 0x007E, 0x007F,
	/* 0x080 */ 0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087, 0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F,
	/* 0x090 */ 0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097, 0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F,
	/* 0x0A0 */ 0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, 0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
	/* 0x0B0 */ 0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x03BC, 0x00B6, 0x00B7, 0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
	/* 0x0C0 */ 0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7, 0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
	/* 0x0D0 */ 0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00D7, 0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00DF,
	/* 0x0E0 */ 0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7, 0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
	/* 0x0F0 */ 0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7, 0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF,
	/* 0x100 */ 0x0101, 0x0101, 0x0103, 0x0103, 0x0105, 0x0105, 0x0107, 0x0107, 0x0109, 0x0109, 0x010B, 0x010B, 0x010D, 0x010D, 0x010F, 0x010F,
	/* 0x110 */ 0x0111, 0x0111, 0x0113, 0x0113, 0x0115, 0x0115, 0x0117, 0x0117, 0x0119, 0x0119, 0x011B, 0x011B, 0x011D, 0x011D, 0x011F, 0x011F,
	/* 0x120 */ 0x0121, 0x0121, 0x0123, 0x0123, 0x0125, 0x0125, 0x0127, 0x0127, 0x0129, 0x0129, 0x012B, 0x012B, 0x012D, 0x012D, 0x012F, 0x012F,
	/* 0x130 */ 0x0130, 0x0131, 0x0133, 0x0133, 0x0135, 0x0135, 0x0137, 0x0137, 0x0138, 0x013A, 0x013A, 0x013C, 0x013C, 0x013E, 0x013E, 0x0140,
	/* 0x140 */ 0x0140, 0x0142, 0x0142, 0x0144, 0x0144, 0x0146, 0x0146, 0x0148, 0x0148, 0x0149, 0x014B, 0x014B, 0x014D, 0x014D, 0x014F, 0x014F,
	/* 0x150 */ 0x0151, 0x0151, 0x0153, 0x0153, 0x0155, 0x0155, 0x0157, 0x0157, 0x0159, 0x0159, 0x015B, 0x015B, 0x015D, 0x015D, 0x015F, 0x015F,
	/* 0x160 */ 0x0161, 0x0161, 0x0163, 0x0163, 0x0165, 0x0165, 0x0167, 0x0167, 0x0169, 0x0169, 0x016B, 0x016B, 0x016D, 0x016D, 0x016F, 0x016F,
	/* 0x170 */ 0x0171, 0x0171, 0x0173, 0x0173, 0x0175, 0x0175, 0x0177, 0x0177, 0x00FF, 0x017A, 0x017A, 0x017C, 0x017C, 0x017E, 0x017E, 0x0073,
	/* 0x180 */ 0x0180, 0x0253, 0x0183, 0x0183, 0x0185, 0x0185, 0x0254, 0x0188, 0x0188, 0x0256, 0x0257, 0x018C, 0x018C, 0x018D, 0x01DD, 0x0259,
	/* 0x190 */ 0x025B, 0x0192, 0x0192, 0x0260, 0x0263, 0x0195, 0x0269, 0x0268, 0x0199, 0x0199, 0x019A, 0x019B, 0x026F, 0x0272, 0x019E, 0x0275,
	/* 0x1A0 */ 0x01A1, 0x01A1, 0x01A3, 0x01A3, 0x01A5, 0x01A5, 0x0280, 0x01A8, 0x01A8, 0x0283, 0x01AA, 0x01AB, 0x01AD, 0x01AD, 0x0288, 0x01B0,
	/* 0x1B0 */ 0x01B0, 0x028A, 0x028B, 0x01B4, 0x01B4, 0x01B6, 0x01B6, 0x0292, 0x01B9, 0x01B9, 0x01BA, 0x01BB, 0x01BD, 0x01BD, 0x01BE, 0x01BF,
	/* 0x1C0 */ 0x01C0, 0x01C1, 0x01C2, 0x01C3, 0x01C6, 0x01C6, 0x01C6, 0x01C9, 0x01C9, 0x01C9, 0x01CC, 0x01CC, 0x01CC, 0x01CE, 0x01CE, 0x01D0,
	/* 0x1D0 */ 0x01D0, 0x01D2, 0x01D2, 0x01D4, 0x01D4, 0x01D6, 0x01D6, 0x01D8, 0x01D8, 0x01DA, 0x01DA, 0x01DC, 0x01DC, 0x01DD, 0x01DF, 0x01DF,
	/* 0x1E0 */ 0x01E1, 0x01E1, 0x01E3, 0x01E3, 0x01E5, 0x01E5, 0x01E7, 0x01E7, 0x01E9, 0x01E9, 0x01EB, 0x01EB, 0x01ED, 0x01ED, 0x01EF, 0x01EF,
	/* 0x1F0 */ 0x01F0, 0x01F3, 0x01F3, 0x01F3, 0x01F5, 0x01F5, 0x0195, 0x01BF, 0x01F9, 0x01F9, 0x01FB, 0x01FB, 0x01FD, 0x01FD, 0x01FF, 0x01FF,
	/* 0x200 */ 0x0201, 0x0201, 0x0203, 0x0203, 0x0205, 0x0205, 0x0207, 0x0207, 0x0209, 0x0209, 0x020B, 0x020B, 0x020D, 0x020D, 0x020F, 0x020F,
	/* 0x210 */ 0x0211, 0x0211, 0x0213, 0x0213, 0x0215, 0x0215, 0x0217, 0x0217, 0x0219, 0x0219, 0x021B, 0x021B, 0x021D, 0x021D, 0x021F, 0x021F,
	/* 0x220 */ 0x019E, 0x0221, 0x0223, 0x0223, 0x0225, 0x0225, 0x0227, 0x0227, 0x0229, 0x0229, 0x022B, 0x022B, 0x022D, 0x022D, 0x022F, 0x022F,
	/* 0x230 */ 0x0231, 0x0231, 0x0233, 0x0233, 0x0234, 0x0235, 0x0236, 0x0237, 0x0238, 0x0239, 0x2C65, 0x023C, 0x023C, 0x019A, 0x2C66, 0x023F,
	/* 0x240 */ 0x0240, 0x0242, 0x0242, 0x0180, 0x0289, 0x028C, 0x0247, 0x0247, 0x0249, 0x0249, 0x024B, 0x024B, 0x024D, 0x024D, 0x024F, 0x024F,
	/* 0x250 */ 0x0250, 0x0251, 0x0252, 0x0253, 0x0254, 0x0255, 0x0256, 0x0257, 0x0258, 0x0259, 0x025A, 0x025B, 0x025C, 0x025D, 0x025E, 0x025F,
	/* 0x260 */ 0x0260, 0x0261, 0x0262, 0x0263, 0x0264, 0x0265, 0x0266, 0x0267, 0x0268, 0x0269, 0x026A, 0x026B, 0x026C, 0x026D, 0x026E, 0x026F,
	/* 0x270 */ 0x0270, 0x0271, 0x0272, 0x0273, 0x0274, 0x0275, 0x0276, 0x0277, 0x0278, 0x0279, 0x027A, 0x027B, 0x027C, 0x027D, 0x027E, 0x027F,
	/* 0x280 */ 0x0280, 0x0281, 0x0282, 0x0283, 0x0284, 0x0285, 0x0286, 0x0287, 0x0288, 0x0289, 0x028A, 0x028B, 0x028C, 0x028D, 0x028E, 0x028F,
	/* 0x290 */ 0x0290, 0x0291, 0x0292, 0x0293, 0x0294, 0x0295, 0x0296, 0x0297, 0x0298, 0x0299, 0x029A, 0x029B, 0x029C, 0x029D, 0x029E, 0x029F,
	/* 0x2A0 */ 0x02A0, 0x02A1, 0x02A2, 0x02A3, 0x02A4, 0x02A5, 0x02A6, 0x02A7, 0x02A8, 0x02A9, 0x02AA, 0x02AB, 0x02AC, 0x02AD, 0x02AE, 0x02AF,
	/* 0x2B0 */ 0x02B0, 0x02B1, 0x02B2, 0x02B3, 0x02B4, 0x02B5, 0x02B6, 0x02B7, 0x02B8, 0x02B9, 0x02BA, 0x02BB, 0x02BC, 0x02BD, 0x02BE, 0x02BF,
	/* 0x2C0 */ 0x02C0, 0x02C1, 0x02C2, 0x02C3, 0x02C4, 0x02C5, 0x02C6, 0x02C7, 0x02C8, 0x02C9, 0x02CA, 0x02CB, 0x02CC, 0x02CD, 0x02CE, 0x02CF,
	/* 0x2D0 */ 0x02D0, 0x02D1, 0x02D2, 0x02D3, 0x02D4, 0x02D5, 0x02D6, 0x02D7, 0x02D8, 0x02D9, 0x02DA, 0x02DB, 0x02DC, 0x02DD, 0x02DE, 0x02DF,
	/* 0x2E0 */ 0x02E0, 0x02E1, 0x02E2, 0x02E3, 0x02E4, 0x02E5, 0x02E6, 0x02E7, 0x02E8, 0x02E9, 0x02EA, 0x02EB, 0x02EC, 0x02ED, 0x02EE, 0x02EF,
	/* 0x2F0 */ 0x02F0, 0x02F1, 0x02F2, 0x02F3, 0x02F4, 0x02F5, 0x02F6, 0x02F7, 0x02F8, 0x02F9, 0x02FA, 0x02FB, 0x02FC, 0x02FD, 0x02FE, 0x02FF,
	/* 0x300 */ 0x0300, 0x0301, 0x0302, 0x0303, 0x0304, 0x0305, 0x0306, 0x0307, 0x0308, 0x0309, 0x030A, 0x030B, 0x030C, 0x030D, 0x030E, 0x030F,
	/* 0x310 */ 0x0310, 0x0311, 0x0312, 0x0313, 0x0314, 0x0315, 0x0316, 0x0317, 0x0318, 0x0319, 0x031A, 0x031B, 0x031C, 0x031D, 0x031E, 0x031F,
	/* 0x320 */ 0x0320, 0x0321, 0x0322, 0x0323, 0x0324, 0x0325, 0x0326, 0x0327, 0x0328, 0x0329, 0x032A, 0x032B, 0x032C, 0x032D, 0x032E, 0x032F,
	/* 0x330 */ 0x0330, 0x0331, 0x0332, 0x0333, 0x0334, 0x0335, 0x0336, 0x0337, 0x0338, 0x0339, 0x033A, 0x033B, 0x033C, 0x033D, 0x033E, 0x033F,
	/* 0x340 */ 0x0340, 0x0341, 0x0342, 0x0343, 0x0344, 0x03B9, 0x0346, 0x0347, 0x0348, 0x0349, 0x034A, 0x034B, 0x034C, 0x034D, 0x034E, 0x034F,
	/* 0x350 */ 0x0350, 0x0351, 0x0352, 0x0353, 0x0354, 0x0355, 0x0356, 0x0357, 0x0358, 0x0359, 0x035A, 0x035B, 0x035C, 0x035D, 0x035E, 0x035F,
	/* 0x360 */ 0x0360, 0x0361, 0x0362, 0x0363, 0x0364, 0x0365, 0x0366, 0x0367, 0x0368, 0x0369, 0x036A, 0x036B, 0x036C, 0x036D, 0x036E, 0x036F,
	/* 0x370 */ 0x0371, 0x0371, 0x0373, 0x0373, 0x0374, 0x0375, 0x0377, 0x0377, 0x0378, 0x0379, 0x037A, 0x037B, 0x037C, 0x037D, 0x037E, 0x03F3,
	/* 0x380 */ 0x0380, 0x0381, 0x0382, 0x0383, 0x0384, 0x0385, 0x03AC, 0x0387, 0x03AD, 0x03AE, 0x03AF, 0x038B, 0x03CC, 0x038D, 0x03CD, 0x03CE,
	/* 0x390 */ 0x0390, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, 0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF,
	/* 0x3A0 */ 0x03C0, 0x03C1, 0x03A2, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7, 0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03AC, 0x03AD, 0x03AE, 0x03AF,
	/* 0x3B0 */ 0x03B0, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, 0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF,
	/* 0x3C0 */ 0x03C0, 0x03C1, 0x03C3, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7, 0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03CC, 0x03CD, 0x03CE, 0x03D7,
	/* 0x3D0 */ 0x03B2, 0x03B8, 0x03D2, 0x03D3, 0x03D4, 0x03C6, 0x03C0, 0x03D7, 0x03D9, 0x03D9, 0x03DB, 0x03DB, 0x03DD, 0x03DD, 0x03DF, 0x03DF,
	/* 0x3E0 */ 0x03E1, 0x03E1, 0x03E3, 0x03E3, 0x03E5, 0x03E5, 0x03E7, 0x03E7, 0x03E9, 0x03E9, 0x03EB, 0x03EB, 0x03ED, 0x03ED, 0x03EF, 0x03EF,
	/* 0x3F0 */ 0x03BA, 0x03C1, 0x03F2, 0x03F3, 0x03B8, 0x03B5, 0x03F6, 0x03F8, 0x03F8, 0x03F2, 0x03FB, 0x03FB, 0x03FC, 0x037B, 0x037C, 0x037D,
	/* 0x400 */ 0x0450, 0x0451, 0x0452, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457, 0x0458, 0x0459, 0x045A, 0x045B, 0x045C, 0x045D, 0x045E, 0x045F,
	/* 0x410 */ 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F,
	/* 0x420 */ 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, 0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F,
	/* 0x430 */ 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F,
	/* 0x440 */ 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, 0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F,
	/* 0x450 */ 0x0450, 0x0451, 0x0452, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457, 0x0458, 0x0459, 0x045A, 0x045B, 0x045C, 0x045D, 0x045E, 0x045F,
	/* 0x460 */ 0x0461, 0x0461, 0x0463, 0x0463, 0x0465, 0x0465, 0x0467, 0x0467, 0x0469, 0x0469, 0x046B, 0x046B, 0x046D, 0x046D, 0x046F, 0x046F,
	/* 0x470 */ 0x0471, 0x0471, 0x0473, 0x0473, 0x0475, 0x0475, 0x0477, 0x0477, 0x0479, 0x0479, 0x047B, 0x047B, 0x047D, 0x047D, 0x047F, 0x047F,
	/* 0x480 */ 0x0481, 0x0481, 0x0482, 0x0483, 0x0484, 0x0485, 0x0486, 0x0487, 0x0488, 0x0489, 0x048B, 0x048B, 0x048D, 0x048D, 0x048F, 0x048F,
	/* 0x490 */ 0x0491, 0x0491, 0x0493, 0x0493, 0x0495, 0x0495, 0x0497, 0x0497, 0x0499, 0x0499, 0x049B, 0x049B, 0x049D, 0x049D, 0x049F, 0x049F,
	/* 0x4A0 */ 0x04A1, 0x04A1, 0x04A3, 0x04A3, 0x04A5, 0x04A5, 0x04A7, 0x04A7, 0x04A9, 0x04A9, 0x04AB, 0x04AB, 0x04AD, 0x04AD, 0x04AF, 0x04AF,
	/* 0x4B0 */ 0x04B1, 0x04B1, 0x04B3, 0x04B3, 0x04B5, 0x04B5, 0x04B7, 0x04B7, 0x04B9, 0x04B9, 0x04BB, 0x04BB, 0x04BD, 0x04BD, 0x04BF, 0x04BF,
	/* 0x4C0 */ 0x04CF, 0x04C2, 0x04C2, 0x04C4, 0x04C4, 0x04C6, 0x04C6, 0x04C8, 0x04C8, 0x04CA, 0x04CA, 0x04CC, 0x04CC, 0x04CE, 0x04CE, 0x04CF,
	/* 0x4D0 */ 0x04D1, 0x04D1, 0x04D3, 0x04D3, 0x04D5, 0x04D5, 0x04D7, 0x04D7, 0x04D9, 0x04D9, 0x04DB, 0x04DB, 0x04DD, 0x04DD, 0x04DF, 0x04DF,
	/* 0x4E0 */ 0x04E1, 0x04E1, 0x04E3, 0x04E3, 0x04E5, 0x04E5, 0x04E7, 0x04E7, 0x04E9, 0x04E9, 0x04EB, 0x04EB, 0x04ED, 0x04ED, 0x04EF, 0x04EF,
	/* 0x4F0 */ 0x04F1, 0x04F1, 0x04F3, 0x04F3, 0x04F5, 0x04F5, 0x04F7, 0x04F7, 0x04F9, 0x04F9, 0x04FB, 0x04FB, 0x04FD, 0x04FD, 0x04FF, 0x04FF
};
/* End generated data. */

#endif  // #ifndef vfs_unicode_data_h
