// Copyright (c) 2023 Apple Inc. All rights reserved.
//
// @APPLE_OSREFERENCE_LICENSE_HEADER_START@
//
// This file contains Original Code and/or Modifications of Original Code
// as defined in and that are subject to the Apple Public Source License
// Version 2.0 (the 'License'). You may not use this file except in
// compliance with the License. The rights granted to you under the License
// may not be used to create, or enable the creation or redistribution of,
// unlawful or unlicensed copies of an Apple operating system, or to
// circumvent, violate, or enable the circumvention or violation of, any
// terms of an Apple operating system software license agreement.
//
// Please obtain a copy of the License at
// http://www.opensource.apple.com/apsl/ and read it before using this file.
//
// The Original Code and all software distributed under the License are
// distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
// EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
// INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
// Please see the License for the specific language governing rights and
// limitations under the License.
//
// @APPLE_OSREFERENCE_LICENSE_HEADER_END@

#include <arm/cpuid.h>
#include <arm64/cpc_arm64.h>
#include <kern/assert.h>
#include <kern/cpc.h>
#include <kern/startup.h>
#include <stdint.h>
#include <stdbool.h>
#include <stddef.h>

struct cpc_event {
	const char *cev_name;
	uint16_t cev_selector;
};

struct cpc_event_list {
	unsigned int cel_event_count;
	struct cpc_event cel_events[];
};
static const struct cpc_event_list _known_cpmu_events;

static const struct cpc_event_list _no_events = {
	.cel_event_count = 0,
	.cel_events = {},
};

const struct cpc_event_list * const _cpc_known_events[CPC_HW_COUNT] = {
	[CPC_HW_CPMU] = &_known_cpmu_events,
	[CPC_HW_UPMU] = &_no_events,
};

static const struct cpc_event *
_cpc_select_event(cpc_hw_t hw, uint16_t selector)
{
	assert(hw < CPC_HW_COUNT);
	const struct cpc_event_list *list = _cpc_known_events[hw];
	for (unsigned int i = 0; i < list->cel_event_count; i++) {
		if (list->cel_events[i].cev_selector == selector) {
			return &list->cel_events[i];
		}
	}
	return NULL;
}

static
#if !CPC_INSECURE
const
#endif // !CPC_INSECURE
cpc_event_policy_t _cpc_event_policy = CPC_EVPOL_DEFAULT;

cpc_event_policy_t
cpc_get_event_policy(void)
{
	return _cpc_event_policy;
}

#if CPC_INSECURE

void
cpc_set_event_policy(cpc_event_policy_t new_policy)
{
	_cpc_event_policy = new_policy;
}

#endif // CPC_INSECURE

bool
cpc_event_allowed(
	cpc_hw_t hw,
	uint16_t event_selector)
{
	if (event_selector == 0) {
		return true;
	}
	switch (_cpc_event_policy) {
#if CPC_INSECURE
	case CPC_EVPOL_ALLOW_ALL:
		return true;
#endif // CPC_INSECURE
	case CPC_EVPOL_DENY_ALL:
		return false;
	case CPC_EVPOL_RESTRICT_TO_KNOWN:
		return _cpc_select_event(hw, event_selector) != NULL;
	}
	return false;
}


static const struct cpc_event_list _known_cpmu_events = {
#if   defined(ARM64_BOARD_CONFIG_T6000)
	.cel_event_count = 59,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0001, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x0002, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0004, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0005, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0007, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0008, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x000a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x000b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x0051, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x0052, .cev_name = "SCHEDULE_UOP" },
		{ .cev_selector = 0x006c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0070, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0075, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0076, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x007c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x007d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x007e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0084, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x008c, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x008d, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x008e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x008f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0090, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0093, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0095, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0096, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0097, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0098, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0099, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x009a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x009b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x009c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x00a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x00a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x00a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x00a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x00a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x00a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x00a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x00b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x00b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x00b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x00b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x00bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x00c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x00c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x00c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x00c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00cb, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x00d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x00db, .cev_name = "L1I_CACHE_MISS_DEMAND" },
		{ .cev_selector = 0x00de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x00e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x00e6, .cev_name = "LD_NT_UOP" },
	},
#elif defined(ARM64_BOARD_CONFIG_T6020)
	.cel_event_count = 59,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0001, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x0002, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0004, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0005, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0007, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0008, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x000a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x000b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x0051, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x006c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0070, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0075, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0076, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x007c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x007d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x007e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0084, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x008c, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x008d, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x008e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x008f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0090, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0093, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0095, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0096, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0097, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0098, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0099, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x009a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x009b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x009c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x009f, .cev_name = "INST_SIMD_ALU_VEC" },
		{ .cev_selector = 0x00a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x00a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x00a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x00a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x00a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x00a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x00a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x00b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x00b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x00b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x00b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x00bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x00c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x00c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x00c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x00c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00cb, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x00d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x00db, .cev_name = "L1I_CACHE_MISS_DEMAND" },
		{ .cev_selector = 0x00de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x00e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x00e6, .cev_name = "LD_NT_UOP" },
	},
#elif defined(ARM64_BOARD_CONFIG_T6030)
	.cel_event_count = 65,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0001, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x0002, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0084, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x008c, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x008d, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x008e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x008f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0090, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0093, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0095, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0096, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0097, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0098, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0099, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x009a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x009b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x009c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x009f, .cev_name = "INST_SIMD_ALU_VEC" },
		{ .cev_selector = 0x00bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x00c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x00c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x00c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x00c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00cb, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x0182, .cev_name = "MAP_DISPATCH_BUBBLE_IC" },
		{ .cev_selector = 0x0183, .cev_name = "MAP_DISPATCH_BUBBLE_ITLB" },
		{ .cev_selector = 0x01d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x01d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x01db, .cev_name = "L1I_CACHE_MISS_DEMAND" },
		{ .cev_selector = 0x01de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x0269, .cev_name = "MAP_UOP" },
		{ .cev_selector = 0x026c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0270, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0275, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0276, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x027c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x027d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x027e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0283, .cev_name = "SCHEDULE_UOP_ANY" },
		{ .cev_selector = 0x0290, .cev_name = "LDST_UNIT_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0291, .cev_name = "LDST_UNIT_WAITING_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0351, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x0404, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0405, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0407, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0408, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x040a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x040b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x05a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x05a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x05a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x05a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x05a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x05a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x05a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x05b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x05b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x05b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x05b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x05e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x05e6, .cev_name = "LD_NT_UOP" },
	},
#elif defined(ARM64_BOARD_CONFIG_T6031)
	.cel_event_count = 65,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0001, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x0002, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0084, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x008c, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x008d, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x008e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x008f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0090, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0093, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0095, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0096, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0097, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0098, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0099, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x009a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x009b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x009c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x009f, .cev_name = "INST_SIMD_ALU_VEC" },
		{ .cev_selector = 0x00bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x00c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x00c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x00c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x00c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00cb, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x0182, .cev_name = "MAP_DISPATCH_BUBBLE_IC" },
		{ .cev_selector = 0x0183, .cev_name = "MAP_DISPATCH_BUBBLE_ITLB" },
		{ .cev_selector = 0x01d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x01d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x01db, .cev_name = "L1I_CACHE_MISS_DEMAND" },
		{ .cev_selector = 0x01de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x0269, .cev_name = "MAP_UOP" },
		{ .cev_selector = 0x026c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0270, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0275, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0276, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x027c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x027d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x027e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0283, .cev_name = "SCHEDULE_UOP_ANY" },
		{ .cev_selector = 0x0290, .cev_name = "LDST_UNIT_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0291, .cev_name = "LDST_UNIT_WAITING_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0351, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x0404, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0405, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0407, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0408, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x040a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x040b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x05a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x05a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x05a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x05a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x05a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x05a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x05a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x05b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x05b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x05b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x05b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x05e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x05e6, .cev_name = "LD_NT_UOP" },
	},
#elif defined(ARM64_BOARD_CONFIG_T6041)
	.cel_event_count = 102,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0003, .cev_name = "ARM_L1D_CACHE_REFILL" },
		{ .cev_selector = 0x0004, .cev_name = "ARM_L1D_CACHE" },
		{ .cev_selector = 0x0008, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x0010, .cev_name = "ARM_BR_MIS_PRED" },
		{ .cev_selector = 0x0011, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0012, .cev_name = "ARM_BR_PRED" },
		{ .cev_selector = 0x0021, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x0022, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x0023, .cev_name = "ARM_STALL_FRONTEND" },
		{ .cev_selector = 0x0024, .cev_name = "ARM_STALL_BACKEND" },
		{ .cev_selector = 0x0039, .cev_name = "ARM_L1D_CACHE_LMISS_RD" },
		{ .cev_selector = 0x003a, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x003b, .cev_name = "MAP_UOP" },
		{ .cev_selector = 0x003c, .cev_name = "ARM_STALL" },
		{ .cev_selector = 0x003d, .cev_name = "ARM_STALL_SLOT_BACKEND" },
		{ .cev_selector = 0x003e, .cev_name = "ARM_STALL_SLOT_FRONTEND" },
		{ .cev_selector = 0x003f, .cev_name = "ARM_STALL_SLOT" },
		{ .cev_selector = 0x0040, .cev_name = "ARM_L1D_CACHE_RD" },
		{ .cev_selector = 0x0182, .cev_name = "MAP_DISPATCH_BUBBLE_IC" },
		{ .cev_selector = 0x0183, .cev_name = "MAP_DISPATCH_BUBBLE_ITLB" },
		{ .cev_selector = 0x01d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x01d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x01de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x01e1, .cev_name = "MAP_DISPATCH_BUBBLE_SLOT" },
		{ .cev_selector = 0x026c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0270, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0275, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0276, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x027c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x027d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x027e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0283, .cev_name = "SCHEDULE_UOP_ANY" },
		{ .cev_selector = 0x0285, .cev_name = "MAP_INT_SME_UOP" },
		{ .cev_selector = 0x0286, .cev_name = "SME_ENGINE_SM_ENABLE" },
		{ .cev_selector = 0x0287, .cev_name = "SME_ENGINE_SM_ZA_ENABLE" },
		{ .cev_selector = 0x0288, .cev_name = "SME_ENGINE_ZA_ENABLED_SM_DISABLED" },
		{ .cev_selector = 0x028c, .cev_name = "LDST_UNIT_WAITING_SME_ENGINE_INST_QUEUE_FULL" },
		{ .cev_selector = 0x028e, .cev_name = "SCHEDULE_WAITING_SME_ENGINE_REG_DATA" },
		{ .cev_selector = 0x028f, .cev_name = "LDST_UNIT_WAITING_SME_ENGINE_MEM_DATA" },
		{ .cev_selector = 0x0290, .cev_name = "LDST_UNIT_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0291, .cev_name = "LDST_UNIT_WAITING_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0294, .cev_name = "LD_UNIT_WAITING_YOUNG_L1D_CACHE_MISS" },
		{ .cev_selector = 0x02ad, .cev_name = "MAP_RECOVERY" },
		{ .cev_selector = 0x02ae, .cev_name = "MAP_STALL_NONRECOVERY" },
		{ .cev_selector = 0x0351, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x0404, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0405, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0407, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0408, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x040a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x040b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x0508, .cev_name = "LDST_SME_XPG_UOP" },
		{ .cev_selector = 0x0529, .cev_name = "INST_SME_ENGINE_PACKING_FUSED" },
		{ .cev_selector = 0x052c, .cev_name = "LD_BLOCKED_BY_SME_LDST" },
		{ .cev_selector = 0x052e, .cev_name = "ST_BARRIER_BLOCKED_BY_SME_LDST" },
		{ .cev_selector = 0x0573, .cev_name = "LD_SME_NT_UOP" },
		{ .cev_selector = 0x0574, .cev_name = "ST_SME_NT_UOP" },
		{ .cev_selector = 0x0575, .cev_name = "LD_SME_NORMAL_UOP" },
		{ .cev_selector = 0x0576, .cev_name = "ST_SME_NORMAL_UOP" },
		{ .cev_selector = 0x0577, .cev_name = "LDST_SME_PRED_INACTIVE" },
		{ .cev_selector = 0x05a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x05a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x05a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x05a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x05a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x05a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x05a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x05b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x05b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x05b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x05b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x05e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x05e6, .cev_name = "LD_NT_UOP" },
		{ .cev_selector = 0x0884, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x088e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x088f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0890, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0893, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0894, .cev_name = "INST_BRANCH_COND" },
		{ .cev_selector = 0x0895, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0896, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0897, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0898, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0899, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x089a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x089b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x089c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x089f, .cev_name = "INST_SIMD_ALU_VEC" },
		{ .cev_selector = 0x08a0, .cev_name = "INST_SME_ENGINE_SCALARFP" },
		{ .cev_selector = 0x08a1, .cev_name = "INST_SME_ENGINE_LD" },
		{ .cev_selector = 0x08a2, .cev_name = "INST_SME_ENGINE_ST" },
		{ .cev_selector = 0x08a3, .cev_name = "INST_SME_ENGINE_ALU" },
		{ .cev_selector = 0x08bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x08c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x08c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x08c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x08c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x08c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x08c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x08ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x4006, .cev_name = "L1I_CACHE_MISS_DEMAND" },
	},
#elif defined(ARM64_BOARD_CONFIG_T8101)
	.cel_event_count = 59,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0001, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x0002, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0004, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0005, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0007, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0008, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x000a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x000b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x0051, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x0052, .cev_name = "SCHEDULE_UOP" },
		{ .cev_selector = 0x006c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0070, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0075, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0076, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x007c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x007d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x007e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0084, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x008c, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x008d, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x008e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x008f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0090, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0093, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0095, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0096, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0097, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0098, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0099, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x009a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x009b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x009c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x00a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x00a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x00a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x00a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x00a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x00a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x00a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x00b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x00b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x00b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x00b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x00bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x00c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x00c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x00c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x00c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00cb, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x00d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x00db, .cev_name = "L1I_CACHE_MISS_DEMAND" },
		{ .cev_selector = 0x00de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x00e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x00e6, .cev_name = "LD_NT_UOP" },
	},
#elif defined(ARM64_BOARD_CONFIG_T8103)
	.cel_event_count = 59,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0001, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x0002, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0004, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0005, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0007, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0008, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x000a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x000b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x0051, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x0052, .cev_name = "SCHEDULE_UOP" },
		{ .cev_selector = 0x006c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0070, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0075, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0076, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x007c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x007d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x007e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0084, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x008c, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x008d, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x008e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x008f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0090, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0093, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0095, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0096, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0097, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0098, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0099, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x009a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x009b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x009c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x00a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x00a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x00a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x00a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x00a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x00a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x00a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x00b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x00b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x00b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x00b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x00bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x00c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x00c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x00c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x00c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00cb, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x00d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x00db, .cev_name = "L1I_CACHE_MISS_DEMAND" },
		{ .cev_selector = 0x00de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x00e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x00e6, .cev_name = "LD_NT_UOP" },
	},
#elif defined(ARM64_BOARD_CONFIG_T8112)
	.cel_event_count = 59,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0001, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x0002, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0004, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0005, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0007, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0008, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x000a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x000b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x0051, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x006c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0070, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0075, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0076, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x007c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x007d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x007e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0084, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x008c, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x008d, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x008e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x008f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0090, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0093, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0095, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0096, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0097, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0098, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0099, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x009a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x009b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x009c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x009f, .cev_name = "INST_SIMD_ALU_VEC" },
		{ .cev_selector = 0x00a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x00a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x00a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x00a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x00a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x00a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x00a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x00b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x00b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x00b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x00b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x00bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x00c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x00c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x00c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x00c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00cb, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x00d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x00db, .cev_name = "L1I_CACHE_MISS_DEMAND" },
		{ .cev_selector = 0x00de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x00e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x00e6, .cev_name = "LD_NT_UOP" },
	},
#elif defined(ARM64_BOARD_CONFIG_T8122_T8130)
	.cel_event_count = 65,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0001, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x0002, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0084, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x008c, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x008d, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x008e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x008f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0090, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0093, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0095, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0096, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0097, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0098, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0099, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x009a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x009b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x009c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x009f, .cev_name = "INST_SIMD_ALU_VEC" },
		{ .cev_selector = 0x00bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x00c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x00c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x00c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x00c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x00cb, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x0182, .cev_name = "MAP_DISPATCH_BUBBLE_IC" },
		{ .cev_selector = 0x0183, .cev_name = "MAP_DISPATCH_BUBBLE_ITLB" },
		{ .cev_selector = 0x01d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x01d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x01db, .cev_name = "L1I_CACHE_MISS_DEMAND" },
		{ .cev_selector = 0x01de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x0269, .cev_name = "MAP_UOP" },
		{ .cev_selector = 0x026c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0270, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0275, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0276, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x027c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x027d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x027e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0283, .cev_name = "SCHEDULE_UOP_ANY" },
		{ .cev_selector = 0x0290, .cev_name = "LDST_UNIT_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0291, .cev_name = "LDST_UNIT_WAITING_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0351, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x0404, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0405, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0407, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0408, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x040a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x040b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x05a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x05a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x05a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x05a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x05a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x05a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x05a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x05b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x05b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x05b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x05b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x05e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x05e6, .cev_name = "LD_NT_UOP" },
	},
#elif defined(ARM64_BOARD_CONFIG_T8132)
	.cel_event_count = 102,
	.cel_events = {
		{ .cev_selector = 0x0000, .cev_name = "NONE" },
		{ .cev_selector = 0x0003, .cev_name = "ARM_L1D_CACHE_REFILL" },
		{ .cev_selector = 0x0004, .cev_name = "ARM_L1D_CACHE" },
		{ .cev_selector = 0x0008, .cev_name = "INST_ALL" },
		{ .cev_selector = 0x0010, .cev_name = "ARM_BR_MIS_PRED" },
		{ .cev_selector = 0x0011, .cev_name = "CORE_ACTIVE_CYCLE" },
		{ .cev_selector = 0x0012, .cev_name = "ARM_BR_PRED" },
		{ .cev_selector = 0x0021, .cev_name = "INST_BRANCH" },
		{ .cev_selector = 0x0022, .cev_name = "BRANCH_MISPRED_NONSPEC" },
		{ .cev_selector = 0x0023, .cev_name = "ARM_STALL_FRONTEND" },
		{ .cev_selector = 0x0024, .cev_name = "ARM_STALL_BACKEND" },
		{ .cev_selector = 0x0039, .cev_name = "ARM_L1D_CACHE_LMISS_RD" },
		{ .cev_selector = 0x003a, .cev_name = "RETIRE_UOP" },
		{ .cev_selector = 0x003b, .cev_name = "MAP_UOP" },
		{ .cev_selector = 0x003c, .cev_name = "ARM_STALL" },
		{ .cev_selector = 0x003d, .cev_name = "ARM_STALL_SLOT_BACKEND" },
		{ .cev_selector = 0x003e, .cev_name = "ARM_STALL_SLOT_FRONTEND" },
		{ .cev_selector = 0x003f, .cev_name = "ARM_STALL_SLOT" },
		{ .cev_selector = 0x0040, .cev_name = "ARM_L1D_CACHE_RD" },
		{ .cev_selector = 0x0182, .cev_name = "MAP_DISPATCH_BUBBLE_IC" },
		{ .cev_selector = 0x0183, .cev_name = "MAP_DISPATCH_BUBBLE_ITLB" },
		{ .cev_selector = 0x01d4, .cev_name = "L1I_TLB_MISS_DEMAND" },
		{ .cev_selector = 0x01d6, .cev_name = "MAP_DISPATCH_BUBBLE" },
		{ .cev_selector = 0x01de, .cev_name = "FETCH_RESTART" },
		{ .cev_selector = 0x01e1, .cev_name = "MAP_DISPATCH_BUBBLE_SLOT" },
		{ .cev_selector = 0x026c, .cev_name = "INTERRUPT_PENDING" },
		{ .cev_selector = 0x0270, .cev_name = "MAP_STALL_DISPATCH" },
		{ .cev_selector = 0x0275, .cev_name = "MAP_REWIND" },
		{ .cev_selector = 0x0276, .cev_name = "MAP_STALL" },
		{ .cev_selector = 0x027c, .cev_name = "MAP_INT_UOP" },
		{ .cev_selector = 0x027d, .cev_name = "MAP_LDST_UOP" },
		{ .cev_selector = 0x027e, .cev_name = "MAP_SIMD_UOP" },
		{ .cev_selector = 0x0283, .cev_name = "SCHEDULE_UOP_ANY" },
		{ .cev_selector = 0x0285, .cev_name = "MAP_INT_SME_UOP" },
		{ .cev_selector = 0x0286, .cev_name = "SME_ENGINE_SM_ENABLE" },
		{ .cev_selector = 0x0287, .cev_name = "SME_ENGINE_SM_ZA_ENABLE" },
		{ .cev_selector = 0x0288, .cev_name = "SME_ENGINE_ZA_ENABLED_SM_DISABLED" },
		{ .cev_selector = 0x028c, .cev_name = "LDST_UNIT_WAITING_SME_ENGINE_INST_QUEUE_FULL" },
		{ .cev_selector = 0x028e, .cev_name = "SCHEDULE_WAITING_SME_ENGINE_REG_DATA" },
		{ .cev_selector = 0x028f, .cev_name = "LDST_UNIT_WAITING_SME_ENGINE_MEM_DATA" },
		{ .cev_selector = 0x0290, .cev_name = "LDST_UNIT_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0291, .cev_name = "LDST_UNIT_WAITING_OLD_L1D_CACHE_MISS" },
		{ .cev_selector = 0x0294, .cev_name = "LD_UNIT_WAITING_YOUNG_L1D_CACHE_MISS" },
		{ .cev_selector = 0x02ad, .cev_name = "MAP_RECOVERY" },
		{ .cev_selector = 0x02ae, .cev_name = "MAP_STALL_NONRECOVERY" },
		{ .cev_selector = 0x0351, .cev_name = "SCHEDULE_EMPTY" },
		{ .cev_selector = 0x0404, .cev_name = "L1I_TLB_FILL" },
		{ .cev_selector = 0x0405, .cev_name = "L1D_TLB_FILL" },
		{ .cev_selector = 0x0407, .cev_name = "MMU_TABLE_WALK_INSTRUCTION" },
		{ .cev_selector = 0x0408, .cev_name = "MMU_TABLE_WALK_DATA" },
		{ .cev_selector = 0x040a, .cev_name = "L2_TLB_MISS_INSTRUCTION" },
		{ .cev_selector = 0x040b, .cev_name = "L2_TLB_MISS_DATA" },
		{ .cev_selector = 0x0508, .cev_name = "LDST_SME_XPG_UOP" },
		{ .cev_selector = 0x0529, .cev_name = "INST_SME_ENGINE_PACKING_FUSED" },
		{ .cev_selector = 0x052c, .cev_name = "LD_BLOCKED_BY_SME_LDST" },
		{ .cev_selector = 0x052e, .cev_name = "ST_BARRIER_BLOCKED_BY_SME_LDST" },
		{ .cev_selector = 0x0573, .cev_name = "LD_SME_NT_UOP" },
		{ .cev_selector = 0x0574, .cev_name = "ST_SME_NT_UOP" },
		{ .cev_selector = 0x0575, .cev_name = "LD_SME_NORMAL_UOP" },
		{ .cev_selector = 0x0576, .cev_name = "ST_SME_NORMAL_UOP" },
		{ .cev_selector = 0x0577, .cev_name = "LDST_SME_PRED_INACTIVE" },
		{ .cev_selector = 0x05a0, .cev_name = "L1D_TLB_ACCESS" },
		{ .cev_selector = 0x05a1, .cev_name = "L1D_TLB_MISS" },
		{ .cev_selector = 0x05a2, .cev_name = "L1D_CACHE_MISS_ST" },
		{ .cev_selector = 0x05a3, .cev_name = "L1D_CACHE_MISS_LD" },
		{ .cev_selector = 0x05a6, .cev_name = "LD_UNIT_UOP" },
		{ .cev_selector = 0x05a7, .cev_name = "ST_UNIT_UOP" },
		{ .cev_selector = 0x05a8, .cev_name = "L1D_CACHE_WRITEBACK" },
		{ .cev_selector = 0x05b1, .cev_name = "LDST_X64_UOP" },
		{ .cev_selector = 0x05b2, .cev_name = "LDST_XPG_UOP" },
		{ .cev_selector = 0x05b3, .cev_name = "ATOMIC_OR_EXCLUSIVE_SUCC" },
		{ .cev_selector = 0x05b4, .cev_name = "ATOMIC_OR_EXCLUSIVE_FAIL" },
		{ .cev_selector = 0x05e5, .cev_name = "ST_NT_UOP" },
		{ .cev_selector = 0x05e6, .cev_name = "LD_NT_UOP" },
		{ .cev_selector = 0x0884, .cev_name = "FLUSH_RESTART_OTHER_NONSPEC" },
		{ .cev_selector = 0x088e, .cev_name = "INST_BRANCH_CALL" },
		{ .cev_selector = 0x088f, .cev_name = "INST_BRANCH_RET" },
		{ .cev_selector = 0x0890, .cev_name = "INST_BRANCH_TAKEN" },
		{ .cev_selector = 0x0893, .cev_name = "INST_BRANCH_INDIR" },
		{ .cev_selector = 0x0894, .cev_name = "INST_BRANCH_COND" },
		{ .cev_selector = 0x0895, .cev_name = "INST_INT_LD" },
		{ .cev_selector = 0x0896, .cev_name = "INST_INT_ST" },
		{ .cev_selector = 0x0897, .cev_name = "INST_INT_ALU" },
		{ .cev_selector = 0x0898, .cev_name = "INST_SIMD_LD" },
		{ .cev_selector = 0x0899, .cev_name = "INST_SIMD_ST" },
		{ .cev_selector = 0x089a, .cev_name = "INST_SIMD_ALU" },
		{ .cev_selector = 0x089b, .cev_name = "INST_LDST" },
		{ .cev_selector = 0x089c, .cev_name = "INST_BARRIER" },
		{ .cev_selector = 0x089f, .cev_name = "INST_SIMD_ALU_VEC" },
		{ .cev_selector = 0x08a0, .cev_name = "INST_SME_ENGINE_SCALARFP" },
		{ .cev_selector = 0x08a1, .cev_name = "INST_SME_ENGINE_LD" },
		{ .cev_selector = 0x08a2, .cev_name = "INST_SME_ENGINE_ST" },
		{ .cev_selector = 0x08a3, .cev_name = "INST_SME_ENGINE_ALU" },
		{ .cev_selector = 0x08bf, .cev_name = "L1D_CACHE_MISS_LD_NONSPEC" },
		{ .cev_selector = 0x08c0, .cev_name = "L1D_CACHE_MISS_ST_NONSPEC" },
		{ .cev_selector = 0x08c1, .cev_name = "L1D_TLB_MISS_NONSPEC" },
		{ .cev_selector = 0x08c4, .cev_name = "ST_MEM_ORDER_VIOL_LD_NONSPEC" },
		{ .cev_selector = 0x08c5, .cev_name = "BRANCH_COND_MISPRED_NONSPEC" },
		{ .cev_selector = 0x08c6, .cev_name = "BRANCH_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x08c8, .cev_name = "BRANCH_RET_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x08ca, .cev_name = "BRANCH_CALL_INDIR_MISPRED_NONSPEC" },
		{ .cev_selector = 0x4006, .cev_name = "L1I_CACHE_MISS_DEMAND" },
	},
#else
	.cel_event_count = 0,
	.cel_events = {
	},
#endif
};
