/*
 * Copyright (c) 2024 Apple Inc. All rights reserved.
 *
 * @APPLE_OSREFERENCE_LICENSE_HEADER_START@
 *
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. The rights granted to you under the License
 * may not be used to create, or enable the creation or redistribution of,
 * unlawful or unlicensed copies of an Apple operating system, or to
 * circumvent, violate, or enable the circumvention or violation of, any
 * terms of an Apple operating system software license agreement.
 *
 * Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 *
 * @APPLE_OSREFERENCE_LICENSE_HEADER_END@
 */

#pragma once

/*
 * try_read_write.h
 *
 * Helper functions for userspace tests to read or write memory and
 * verify that EXC_BAD_ACCESS is or is not generated by that operation.
 *
 * To use these functions in your test you must set additional build options.
 * See target `try_read_write_test` in tests/Makefile for an example.
 */

#include <stdint.h>
#include <stdbool.h>
#include <mach/mach_types.h>

/*
 * Set verbose_exc_helper = true to log exception information with T_LOG().
 * The default is true.
 */
extern bool verbose_exc_helper;

/*
 * Tries to read a single byte from an address.
 * Returns true if the read succeeded.
 * Aborts if an exception other than EXC_BAD_ACCESS is generated.
 * On exit:
 *     *out_byte is the value read, or an indeterminate value if the read failed.
 *     *out_error is the EXC_BAD_ACCESS error code
 *         (typically KERN_PROTECTION_FAILURE or KERN_INVALID_ADDRESS)
 *         or 0 if the read succeeded.
 *
 * To use this function in your test you must set additional build options.
 * See target `try_read_write_test` in tests/Makefile for an example.
 */
extern bool
try_read_byte(
	mach_vm_address_t addr,
	uint8_t * const out_byte,
	kern_return_t * const out_error);

/*
 * Tries to write a single byte to an address.
 * Returns true if the write succeeded.
 * Aborts if an exception other than EXC_BAD_ACCESS is generated.
 * On exit:
 *     *out_error is the EXC_BAD_ACCESS error code
 *         (typically KERN_PROTECTION_FAILURE or KERN_INVALID__ADDRESS)
 *         or 0 if the write succeeded.
 *
 * To use this function in your test you must set additional build options.
 * See target `try_read_write_test` in tests/Makefile for an example.
 */
extern bool
try_write_byte(
	mach_vm_address_t addr,
	uint8_t byte,
	kern_return_t * const out_error);
